package mstar;

import java.util.*;

/**
  *
  * The Chatter class represents output from the collaborative environment, 
  * from either other characters in the environment, the envrionment itself,
  * or the character who executed a command. This class is used to hold
  * the parsed data. In addition a higher level view of the command that
  * caused the data to occur is also provided.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  * @see Connection
  * @see ChatterFilter
  *
  */
public class Chatter {
        /**
          * The source of the chatter is from an undefined command.
          */
	public static final int	UNDEFINED    = -1;
        /**
          * The source of the chatter is from a say command.
          */
	public static final int	SAYS         =  1;
        /**
          * The source of the chatter is from a whisper command.
          */
	public static final int	WHISPERS     =  2;
        /**
          * The source of the chatter is from a page command.
          */
	public static final int	PAGES        =  3;
        /**
          * The source of the chatter is from a mutters command.
          */
	public static final int	MUTTERS      =  4;
        /**
          * The source of the chatter is from a character arriving in the room.
          */
	public static final int ARRIVES      =  5;
        /**
          * The source of the chatter is from a character leaving the room.
          */
	public static final int LEAVES       =  6;
        /**
          * The source of the chatter is from a character executing a pose
	  * pose command.
          */
	public static final int POSES        =  7;
        /**
          * The source of the chatter is from a give command.
          */
	public static final int GIVES_OBJECT =  8;
        /**
          * The source of the chatter is from a command executed by the
	  * character that this class is storing MUD chatter for.
          */
	public static final int MY_ACTION    =  9;
        /**
          * The source of the chatter is from a page command where the 
	  * character that this class is storing MUD chatter for has been
	  * paged to another room.
          */
	public static final int PAGED_TO     = 10;
        /**
          * The source of the chatter is from an emote command.
          */
	public static final int EMOTES       = 11;

	GregorianCalendar	when = null;
	int			how  = UNDEFINED;
	String			what = null;
	String			who  = null;

	/**
	  * Constructs a chatter entry with how undefined, and who, what, and
	  * when set to null.
	  */
	public Chatter() {
		super();
	}

	/**
	  * Constructs a chatter entry with how undefined, and who, and what
	  * set to null, and when set to the specified time the command was
	  * executed.
	  *
	  * @param when The time that the chatter occurred in the collaborative
	  *             environment.
	  */
	public Chatter(GregorianCalendar when) {
		this();
		setWhen(when);
	}

	/**
	  * Constructs a chatter entry with how undefined, what set to null, 
	  * who set to the character in the collaborative environment that
	  * executed the command, and when set to the specified time the
	  * command was executed.
	  *
	  * @param when The time that the chatter occurred in the collaborative
	  *             environment.
	  * @param who  The character in the collaborative environment that
	  *             executed the command.
	  */
	public Chatter(GregorianCalendar when, String who) {
		this(when);
		setWho(who);
	}

	/**
	  * Constructs a chatter entry with how undefined, what set to null, 
	  * who set to the character in the collaborative environment that
	  * executed the command, and when set to the specified time the
	  * command was executed.
	  *
	  * @param when The time that the chatter occurred in the collaborative
	  *             environment.
	  * @param who  The character in the collaborative environment that
	  *             executed the command.
	  * @param how  The type of command that was executed. This can be set
	  *             to any of the fields including: UNDEFINED, SAYS,
	  *             WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  *             GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  */
	public Chatter(GregorianCalendar when, String who, int how) {
		this(when, who);
		setHow(how);
	}

	/**
	  * Constructs a chatter entry with how undefined, what set to either
	  * what the character said or did, who set to the character in the 
	  * collaborative environment that executed the command, and when set 
	  * to the specified time the command was executed.
	  *
	  * @param when The time that the chatter occurred in the collaborative
	  *             environment.
	  * @param who  The character in the collaborative environment that
	  *             executed the command.
	  * @param how  The type of command that was executed. This can be set
	  *             to any of the fields including: UNDEFINED, SAYS,
	  *             WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  *             GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  * @param what What the character said or did. For instance if the
	  *             characters' name is foo and they executed the following 
	  *             command:<P>
	  *                say hello<P>
	  *             Who would be set to foo, what would be set to hello,
	  *             how would be set to SAYS, and when would be set to the
	  *             gregorian calendar with the time the chatter was
	  *             captured.
	  */
	public Chatter(GregorianCalendar when, String who, int how, String what) {
		this(when, who, how);
		setWhat(what);
	}

	/**
	  * Sets the time when the chatter was captured.
	  *
	  * @param when The time that the chatter was captured from the 
	  *             collaborative environment.
	  */
	public void setWhen(GregorianCalendar when) {
		this.when = when;
	}

	/**
	  * Gets the time when the chatter was captured.
	  *
	  * @return a GregorianCalendar with the time that the chatter was
	  *         captured from the collaborative environment.
	  */
	public GregorianCalendar getWhen() {
		return(this.when);
	}

	/**
	  * Sets the type of command that was executed. This can be set
	  * to any of the fields including: UNDEFINED, SAYS,
	  * WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  * GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  *
	  * @param how  The type of command that was executed. This can be set
	  *             to any of the fields including: UNDEFINED, SAYS,
	  *             WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  *             GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  */
	public void setHow(int how) {
		this.how = how;
	}

	/**
	  * Get the type of command that was executed. This can be set
	  * to any of the fields including: UNDEFINED, SAYS,
	  * WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  * GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  *
	  * @return The type of command that was executed. This can be set
	  *         to any of the fields including: UNDEFINED, SAYS,
	  *         WHISPERS, PAGES, MUTTERS, ARRIVES, LEAVES, POSES,
	  *         GIVES_OBJECT, MY_ACTION, PAGED_TO or EMOTES.
	  */
	public int getHow() {
		return(this.how);
	}

	/**
	  * Sets what the character said or did.
	  *
	  * @param what What the character said or did. For instance if the
	  *             characters' name is foo and they executed the following 
	  *             command:<P>
	  *                say hello<P>
	  *             Who would be set to foo, what would be set to hello,
	  *             how would be set to SAYS, and when would be set to the
	  *             gregorian calendar with the time the chatter was
	  *             captured.
	  */
	public void setWhat(String what) {
		this.what = what;
	}

	/**
	  * Gets what the character said or did.
	  *
	  * @return What the character said or did. For instance if the
	  *         characters' name is foo and they executed the following 
	  *         command:<P>
	  *            say hello<P>
	  *         Who would be set to foo, what would be set to hello,
	  *         how would be set to SAYS, and when would be set to the
	  *         gregorian calendar with the time the chatter was
	  *         captured.
	  */
	public String getWhat() {
		return(this.what);
	}

	/**
	  * Sets who executed the command.
	  *
	  * @param who  The character in the collaborative environment that
	  *             executed the command.
	  */
	public void setWho(String who) {
		this.who = who;
	}

	/**
	  * Gets who executed the command.
	  *
	  * @return The character in the collaborative environment that
	  *         executed the command.
	  */
	public String getWho() {
		return(this.who);
	}
}
