/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;
import java.net.*;
import java.util.*;
import gnu.regexp.*;

/**
  * The connection class is an abstract base class that is used to manage the
  * connection for a collaborative environment session.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  */
public abstract class Connection {
	static final boolean debug = true;

	String		hostname      = null;	/* the mud/moo's hostname */
						/* that the robot is to   */
						/* connect to.            */
	int		port          = 4201;	/* the mud/moo's port     */
						/* that the robot is to   */
						/* connect to.            */
	Socket		mudSocket     = null;	/* the socket to talk to  */
						/* the mud/moo on.        */
	PrintStream	out           = null;	/* the output to the      */
						/* socket.                */
	DataInputStream	in            = null;	/* the input from the     */
						/* socket.                */
	String		outputPrefix  = null;	/* the output prefix that */
						/* the mud/moo is going   */
						/* going to send at the   */
						/* beginning of output    */
						/* directed at the robot. */
	String		outputSuffix  = null;	/* the output suffix that */
						/* the mud/moo is going   */
						/* going to send at the   */
						/* end of output          */
	String		robotName     = null;	/* the robots name        */

	/**
	  * Initialize the connection class with a null hostname and set the
	  * port to connect to the collaborative environment at to the
	  * default of 4201.
	  */
	public Connection() {
		super();
		initialize(null, 4201);
	}

	/**
	  * Initialize the connection class with hostname and set the
	  * port to connect to the collaborative environment at to the
	  * default of 4201.
	  *
	  * @param hostname the name of the host to connect to.
	  */
	public Connection(String hostname) {
		super();
		initialize(hostname, 4201);
	}

	/**
	  * Initialize the connection class with hostname and set the
	  * port to connect to the collaborative environment at to the
	  * given port number.
	  *
	  * @param hostname the name of the host to connect to.
	  * @param port the port to connect to.
	  */
	public Connection(String hostname, int port) {
		super();
		initialize(hostname, port);
	}

	/**
	  * Initialize the connection class with a null hostname and set the
	  * port to connect to the collaborative environment at to the
	  * given port number.
	  *
	  * @param port the port to connect to.
	  */
	public Connection(int port) {
		super();
		initialize(null, port);
	}

	void initialize(String hostname, int port) {
		setHostname(hostname);
		setPort(port);
	}

	/**
	  * Sets the hostname to connect to.
	  *
	  * @param hostname the hostname to connect to.
	  */
	public void setHostname(String hostname) {
		this.hostname = hostname;
	}

	/**
	  * Gets the hostname to connect to.
	  *
	  * @return the hostname to connect to.
	  */
	public String getHostname() {
		return(this.hostname);
	}

	/**
	  * Sets the port to connect at.
	  *
	  * @param port the port to connect at.
	  */
	public void setPort(int port) {
		this.port = port;
	}

	/**
	  * Gets the port to connect at.
	  *
	  * @return the port to connect at.
	  */
	public int getPort() {
		return(this.port);
	}

	/**
	  * Gets the stream for receiving data from the collaborative
	  * environment.
	  *
	  * @return the DataInputStream for receiving data from the 
	  * collaborative environment.
	  */
	public DataInputStream getInputStream() {
		return(this.in);
	}

	/**
	  * Gets the stream for sending data to the collaborative
	  * environment.
	  *
	  * @return the PrintStream for sending data to the collaborative 
	  * environment.
	  */
	public PrintStream getOutputStream() {
		return(this.out);
	}

	/**
	  * Sets the string that will be used for the output prefix in commands
	  * returned by the collaborative environment.
	  *
	  * @param outputPrefix the string to use as the output prefix.
	  */
	public void setOutputPrefix(String outputPrefix) {
		this.outputPrefix = outputPrefix;
	}

	/**
	  * Gets the string that will be used for the output prefix in commands
	  * returned by the collaborative environment.
	  *
	  * @return the string to use as the output prefix.
	  */
	public String getOutputPrefix() {
		return(this.outputPrefix);
	}

	/**
	  * Sets the string that will be used for the output suffix in commands
	  * returned by the collaborative environment.
	  *
	  * @param outputSuffix the string to use as the output suffix.
	  */
	public void setOutputSuffix(String outputSuffix) {
		this.outputSuffix = outputSuffix;
	}

	/**
	  * Gets the string that will be used for the output suffix in commands
	  * returned by the collaborative environment.
	  *
	  * @return the string to use as the output suffix.
	  */
	public String getOutputSuffix() {
		return(this.outputSuffix);
	}

	/**
	  * Sets the name of the robot.
	  *
	  * @param robotName the name of the robot.
	  */
	public void setRobotName(String robotName) {
		this.robotName = robotName;
	}

	/**
	  * Gets the name of the robot.
	  *
	  * @return the name of the robot.
	  */
	public String getRobotName() {
		return(this.robotName);
	}

	/**
	  * Find the output prefix in the input stream.
	  */
	public abstract void findOutputPrefix();

	/**
	  * Drain the connection until there is no more output waiting to be
	  * collected. If wait is set to true, wait for output if there is
	  * no output currently waiting.
	  *
	  * @param wait if true, wait for output if none is currently 
	  *             available. If false, and no data is available then
	  *             just return.
	  */
	public abstract void drainConnection(boolean wait);

	/**
	  * Drain the connection until there is no more output waiting to be
	  * collected. Wait for output if there is no output currently waiting.
	  */
	public void drainConnection() {
		drainConnection(true);
	}

	/**
	  * Wait until data becomes available from the collaborative 
	  * environment.
	  */
	public void waitForData() {
		try {
			while (available() == 0) {
				try {
					Thread.sleep(100);
				}
				catch (InterruptedException ie) {
				}
			}
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
	}

	/**
	  * Drain the output from the collaborative environment until the
	  * suffix that was set using setOutputSuffix is found. If wait is
	  * set to true wait until data becomes available from the 
	  * collaborative environment. If wait is set to false and no data is
	  * available return immediately.
	  *
	  * @param wait if true, wait for output if none is currently 
	  *             available. If false, and no data is available then
	  *             just return.
	  */
	public abstract void drainConnectionToSuffix(boolean wait);

	/**
	  * Drain the output from the collaborative environment until the
	  * suffix that was set using setOutputSuffix is found. Wait until data 
	  * becomes available from the collaborative environment.
	  */
	public void drainConnectionToSuffix() {
		drainConnectionToSuffix(true);
	}

	/**
	  * Returns the chatter vector that output from the collaborative
	  * environment is stored in for later processing.
	  *
	  * @return the Vector containing output from the collaborative
	  *         environment.
	  */
	public abstract Vector getChatterVector();

	/**
	  * Clears the chatter vector that output from the collaborative
	  * environment is being stored in.
	  */
	public abstract void clearChatterVector();

	/**
	  *  Connect the robot to the Mud/MOO.
	  */
	public void connectToMud() {
		if (getHostname() != null) {
			try {
				this.mudSocket = new Socket(getHostname(), getPort());
				//this.mudSocket.setKeepAlive(true);
				this.out       = new PrintStream(this.mudSocket.getOutputStream());
				this.in        = new DataInputStream(this.mudSocket.getInputStream());
				drainConnection();
			}
			catch (UnknownHostException uhe) {
				System.exit(-1);
			}
			catch (IOException ioe) {
				System.exit(-1);
			}
		}
	}

	/**
	  * Log the robot into the Mud/MOO. This routine should be called
	  * after connectToMud has been called.
	  *
	  * @param name the name of the robot.
	  * @param password the password of the robot.
	  */
	public abstract void loginToMud(String name, String password);

	/**
	  * Log the robot off the MUD/MOO.
	  */
	public abstract void logOffMud();

	/**
	  * Read a line of output from the MUD/MOO.
	  *
	  * @exception IOException If an I/O error occurs.
	  */
	public String readLine() throws IOException {
		return(this.in.readLine());
	}

	/**
	  * Returns the number of bytes that can be read from the collaborative
	  * environment without blocking.
	  *
	  * @return the number of bytes that can be read from the collaborative
	  *         environment without blocking.
	  * @exception IOException if an I/O error occurs.
	  */
	public int available() throws IOException {
		return(this.in.available());
	}
}
