/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.util.*;
import java.io.*;
import gnu.regexp.*;

/**
  * A class that implements a general map for storing information about each
  * room in the collaborative environment, and a set of methods for searching
  * for rooms, and objects in the map. In addition there are a set of methods
  * for reading and writing the maps to disk in both Java object format and
  * psuedo-MAAS-Neotek format.
  */
public class MudMap extends Vector implements Serializable {
	int	id = 0;
	String	mapFilename  = "tween.map";
	Room	currentRoom  = null;
	Room	previousRoom = null;
	Room	home         = null;

	/**
	  * Create an empty MUD map.
	  */
	public MudMap() {
		super();
	}

	/**
	  * Create a MUD map and add the specified room.
	  *
	  * @param room the room to be added to the MUD map.
	  */
	public MudMap(Room room) {
		super();
		add(room);
	}

	/**
	  * Returns the room identifier for the next room.
	  *
	  * @return the integer identifier for the next room to be added.
	  */
	public synchronized int getId() {
		return(this.id);
	}

	/**
	  * Sets the robot's current room in the MUD.
	  *
	  * @param currentRoom the room that is the robot's current position in
	  *                    the MUD.
	  */
	public void setCurrentRoom(Room currentRoom) {
		this.currentRoom = currentRoom;
	}

	/**
	  * Gets the robot's current room in the MUD.
	  *
	  * @return the room that is the robot's current position in the MUD.
	  */
	public Room getCurrentRoom() {
		return(this.currentRoom);
	}

	/**
	  * Sets the last room that the robot was in.
	  *
	  * @param previousRoom the room that the robot was in before the
	  *                     current room.
	  */
	public void setPreviousRoom(Room previousRoom) {
		this.previousRoom = previousRoom;
	}

	/**
	  * Gets the last room that the robot was in.
	  *
	  * @return the room that the robot was in before the current room.
	  */
	public Room getPreviousRoom() {
		return(this.previousRoom);
	}

	/**
	  * Sets the room that is the robot's home or default room in the
	  * collaborative environment.
	  *
	  * @param home the room that is the robot's home or default room in
	  *             the collaborative environment.
	  */
	public void setHome(Room home) {
		this.home = home;
	}

	/**
	  * Gets the room that is the robot's home or default room in the
	  * collaborative environment.
	  *
	  * @return the room that is the robot's home or default room in the 
	  *         collaborative environment.
	  */
	public Room getHome() {
		return(this.home);
	}

	/**
	  * Sets whether or not to use fallback exits. Fallback exits are used
	  * by the robot when mapping the collaborative environment to help it
	  * locate exits that are not in the obvious exit list.
	  *
	  * @param t if true use fallback exits to locate exits. If false do
	  *          not use the fallback exit list.
	  */
	public void setUseFallbackExits(boolean t) {
		int	i;
		int	length;

		length = this.size();
		for (i = 0; i < length; i++) {
			((Room)elementAt(i)).setUseFallbackExits(t);
		}
	}

	/**
	  * Sets whether or not to use possible exits. Possible exits are used
	  * by the robot when mapping the collaborative environment to help it
	  * locate exits that are not in the obvious exit list. A possible exit
	  * is derived by creating a set of uniterms from the room description.
	  *
	  * @param t if true use possible exits to locate exits. If false do
	  *          not use the possible exit list.
	  */
	public void setUsePossibleExits(boolean t) {
		int	i;
		int	length;

		length = this.size();
		for (i = 0; i < length; i++) {
			((Room)elementAt(i)).setUsePossibleExits(t);
		}
	}

	/**
	  * Sets the list of fallback exits to use.
	  *
	  * @param exits a String array of exit names to use when fallback
	  *              exits are enabled.
	  */
	public void setFallbackExits(String exits[]) {
		int	i;
		int	length;

		length = this.size();
		for (i = 0; i < length; i++) {
			((Room)elementAt(i)).setFallbackExits(exits);
		}
	}

	/**
	  * Loads the MUD map from disk.
	  */
	public static MudMap loadMap() {
		Room	mapAry[]      = null;	/* used by the loadMap    */
						/* method to load the     */
						/* object map from disk.  */
		int	highest = -1;
		MudMap	map     = null;

		try {
			File f = new File("tween.dat");
			if (f.exists()) {
				ObjectInputStream in = new ObjectInputStream(new FileInputStream("tween.dat"));
				mapAry = (Room[])in.readObject();
				in.close();
				map = new MudMap();
				for (int j = 0; j < mapAry.length; j++) {
					if(highest < mapAry[j].getId()) {
						highest = mapAry[j].getId();
					}
					map.fastAdd(mapAry[j]);
				}
			}
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
		catch(ClassNotFoundException cnf) {
			System.out.println(cnf.toString());
		}
		return(map);
	}

	/**
	  * Adds a room to the map and writes the map back to disk.
	  *
	  * @param room the room to add to the the map.
	  */
	public synchronized void add(Room room) {
		room.setId(id);
		id++;
		super.addElement(room);
		WriteMap();
		WriteMapAsObjects();
	}

	/**
	  * Adds a room to the map but does not write the maps to disk.
	  *
	  * @param room the room to add to the map.
	  */
	public synchronized void fastAdd(Room room) {
		room.setId(id);
		id++;
		super.addElement(room);
	}

	/**
	  * Given a room find the same room in the map.
	  *
	  * @param r the room to locate in the map.
	  * @return the room from the map or null if it doesn't exist.
	  */
	public Room getRoom(Room r) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;

		while ((!done) &&
		       (i < length)) {
			if (r.equals((Room)elementAt(i))) {
				room = (Room)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(room);
	}

	/**
	  * Given the id of a room in the map locate and return that room.
	  *
	  * @param id of the room to locate in the map.
	  * @return the room from the map or null if it doesn't exist.
	  */
	public Room getRoom(int id) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;

		while ((!done) &&
		       (i < length)) {
			if (((Room)elementAt(i)).getId() == id) {
				room = (Room)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(room);
	}

	/**
	  * Starting from the beginning of the map, locate a room that has
	  * unexplored exits.
	  *
	  * @return a room with unexplored exits, or null if all exits in all
	  *         rooms have been explored.
	  */
	public Room getRoomWithUnexploredExits() {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room = null;

		while ((!done) &&
		       (i < length)) {
			if (((Room)elementAt(i)).hasUnexploredExits()) {
				room = (Room)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(room);
	}

	/**
	  * Starting from the id greater than after, locate a room that has
	  * unexplored exits.
	  *
	  * @return a room with unexplored exits, or null if all exits in all
	  *         rooms have been explored.
	  */
	public Room getRoomWithUnexploredExits(int after) {
		int	i = after + 1;
		int	length = this.size();
		boolean	done   = false;
		Room	room = null;

		while ((!done) &&
		       (i < length)) {
			if (((Room)elementAt(i)).hasUnexploredExits()) {
				room = (Room)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(room);
	}

	/**
	  * Locate a room in the map with the given name.
	  *
	  * @param name the name of the room to locate.
	  * @return a room with the given name. If two rooms in the 
	  *         collaborative environment have identical names, only the
	  *         first will be returned.
	  */
	public Room getRoom(String name) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;

		while ((!done) &&
		       (i < length)) {
			//room = (Room)elementAt(i);
			if (((Room)elementAt(i)).getName().toLowerCase().equals(name.toLowerCase())) {
				room = (Room)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(room);
	}

	/**
	  * Determines if a room in the map exists with the given name.
	  *
	  * @param name of the room to check for.
	  * @return true if the room exists false if it does not.
	  */
	public boolean roomExists(String name) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			if (((Room)elementAt(i)).getName().equals(name)) {
				room = (Room)elementAt(i);
				rc   = true;
				done = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Determines if a room in the map exists with the given id.
	  *
	  * @param id of the room to check for.
	  * @return true if the room exists false if it does not.
	  */
	public boolean roomExists(int id) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			room = (Room)elementAt(i);
			if (room.getId() == id) {
				rc   = true;
				done = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Determines if a room in the map exists given another room.
	  *
	  * @param theRoom to check the map for.
	  * @return true if the room exists, false if it does not.
	  */
	public boolean roomExists(Room theRoom) {
		int	i      = 0;
		int	length = this.size();
		boolean	done   = false;
		Room	room   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			room = (Room)elementAt(i);
			if (theRoom.equals(room)) {
				done = true;
				rc   = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Returns a string representation of the map.
	  *
	  * @return a string representation of the map.
	  */
	public String toString() {
		StringBuffer	sb;
		int		i;
		Room		aRoom;

		sb = new StringBuffer();
		for (i = 0; i < size(); i++) {
			aRoom = ((Room)elementAt(i));
			sb.append(aRoom.toMap());
		}
		return(sb.toString());
	}

	/**
	  * Clears the visited exits field of each room in the map.
	  */
	public void clearVisitedExits() {
		int	length;
		int	i;
		Room	aRoom;

		length = size();
		for (i = 0; i < length; i++) {
			aRoom = ((Room)elementAt(i));
			aRoom.clearVisitedExits();
		}
	}

	/**
	  *   Set the output filename for the map.
	  */
	public void setMapFilename(String mapFilename) {
		this.mapFilename = mapFilename;
	}

	/**
	  * Gets the output filename for the map.
	  */
	public String getMapFilename() {
		return(this.mapFilename);
	}

	/**
	  * Writes the map out to disk as a set of Java objects.
	  */
	public void WriteMapAsObjects() {
		Room	r[];

		try {
			r = new Room[this.size()];
			this.copyInto(r);
			ObjectOutputStream	out = new ObjectOutputStream(new FileOutputStream("tween.dat"));
			out.writeObject(r);
			out.close();
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
	}

	/**
	  * Writes the map out in pseudo-MAAS-Neotek format.
	  */
	public void WriteMap() {
		FileOutputStream	o = null;
		OutputStreamWriter	w = null;

//System.out.print(".");
		try {
			o = new FileOutputStream(getMapFilename());
			w = new OutputStreamWriter(o);
			w.write(toString());
			w.close();
		}
		catch(IOException ie) {
			System.out.println(ie.toString());
		}
	}

	/**
	  * Finds the next unvisited exit in a room.
	  *
	  * @param r the room to get the next exit from.
	  * @return an Exit that has not been explored, or null if all exits
	  *         in the room have been explored.
	  */
	Exit findNextExit(Room r) {
		boolean	done = false;
		int	i    = 0;
		Exit	rc   = null;
		Exit	g[]  = null;

		g = r.getGoodExits();
		if (g != null) {
			while ((i < g.length) && (!done)) {
				if ((g[i].isConfusing()) ||
				    (((Room)this.elementAt(g[i].getToRoom())).getVisited())) {
					i++;
				}
				else {
					rc = g[i];
					done = true;
				}
			}
		}
		return(rc);
	}

	/**
	  * Clears the visited flag in each of the rooms in the map.
	  */
	public void clearRoomsVisited() {
		int	i;
		int	length;

		length = this.size();
		for (i = 0; i < length; i++) {
			((Room)this.elementAt(i)).setVisited(false);
		}
	}

	/**
	  * Given two rooms return a Stack that has a set of directions to
	  * go from one room to the other.
	  *
	  * @param start the starting room
	  * @param destination the destination room
	  * @return a Stack with a set of directions for navigating between
	  *         the two rooms or null if a path does not exist.
	  */
	public Stack findPath(Room start, Room destination) {
		return(findPath(start.getId(), destination.getId()));
	}

	/**
	  * Given two room identifiers return a Stack that has a set of 
	  * directions to go from one room to the other.
	  *
	  * @param start the identifier of the starting room.
	  * @param destination the identifier of the destination room.
	  * @return a Stack with a set of directions for navigating between
	  *         the two rooms or null if a path does not exist.
	  */
	public Stack findPath(int start, int destination) {
		int		depth       = 0;
		boolean		extended    = true;
		boolean		found       = false;
		int		i;
		PathComponent	path[]      = null;
		int		j;
		Exit		goodExits[] = null;
		Stack		stack       = null;

		// Mass NEOTEK bot method... (adapted for Java)
		path = new PathComponent[this.size()];
		for (i = 0; i < this.size(); i++) {
			path[i] = new PathComponent();
		}
		path[destination].setToRoom(destination);
		path[destination].setDepth(0);
		while ((extended) && (!found) && (depth < this.size())) {
			extended = false;
			i = 0;
			while((i < this.size()) && (!found)) {
				if (path[i].getDepth() < 0) {
					goodExits = getRoom(i).getGoodExits();
					if (goodExits != null) {
						j = 0;
						while ((j < goodExits.length) && (!found)) {
							if ((goodExits[j].getToRoom() != i) &&
							    (path[goodExits[j].getToRoom()].getDepth() == depth)) {
								extended = true;
								path[i].setDirection(goodExits[j].getDirection());
								path[i].setToRoom(goodExits[j].getToRoom());
								path[i].setFromRoom(goodExits[j].getFromRoom());
								path[i].setDepth(path[goodExits[j].getToRoom()].getDepth()+1);
								if (i == start) {
									found = true;
								}
							}
							j++;
						}
					}
				}
				goodExits = null;
				i++;
			}
			depth++;
		}
		if (found) {
			stack = new Stack();
			i = start;
			while (i != destination) {
				stack.push(path[i]);
				i = path[i].getToRoom();
			}
		}
		return(stack);
	}

	/**
	  * Given two room identifiers return a Stack that has a set of 
	  * directions to go from one room to the other. If a path is not
	  * available between the two rooms try finding a path to the room
	  * by going home first.
	  *
	  * @param start the identifier of the starting room.
	  * @param destination the identifier of the destination room.
	  * @return a Stack with a set of directions for navigating between
	  *         the two rooms or null if a path does not exist.
	  */
	public Stack findPathToRoom(int start, int destination) {
		Stack	stack      = null;
		Stack	directions = null;
		Exit	home       = null;

		clearRoomsVisited();
		stack = findPath(start, destination);
		if ((stack == null) ||
		    ((stack != null) && (stack.empty()))) {
			clearRoomsVisited();
			stack = findPath(0, destination);
			if ((stack == null) ||
			    ((stack != null) && (stack.empty()))) {
				//System.out.println("Stack was null or empty!");
				//System.exit(-1);
			}
			else {
				home       = new Exit();
				directions = new Stack();
				while (!stack.empty()) {
					directions.push(stack.pop());
				}
				home.setDirection("home");
				home.setToRoom(((Exit)directions.peek()).getFromRoom());
				directions.push(home);
			}
		}
		else {
			directions = new Stack();
			while (!stack.empty()) {
				directions.push(stack.pop());
			}
		}
		return(directions);
	}

	/**
	  * Given two rooms return a Stack that has a set of directions to
	  * go from one room to the other. If a path is not available between 
	  * the two rooms try finding a path to the room by going home first.
	  *
	  * @param start the starting room
	  * @param destination the destination room
	  * @return a Stack with a set of directions for navigating between
	  *         the two rooms or null if a path does not exist.
	  */
	public Stack findPathToRoom(Room start, Room destination) {
		return(findPathToRoom(start.getId(), destination.getId()));
	}

	/**
	  *  This routine runs through the stack backwards and creates a set of
	  *  human readable directions that the bot is following to go to the
	  *  destination.
	  *
	  *  @param start the index of the room the bot is starting from.
	  *  @param s the stack that has the directions
	  *  @returns human readable instructions
	  */
	public String makeDirections(int start, Stack s) {
		int		i;
		StringBuffer	directions = new StringBuffer();
		Exit		theExit    = null;

		if ((s != null) &&
		    (!s.empty())) {
			directions.append("Starting from ").append(this.getRoom(start).getName()).append("(").append(start).append(")");
			//for (i = 0; i < s.size(); i++) {
			for (i = (s.size() - 1); i >= 0; i--) {
				theExit = ((Exit)s.elementAt(i));
				directions.append(" ").append(theExit.getDirection()).append("[").append(theExit.getFromRoom()).append(", ").append(theExit.getToRoom()).append("]");
			}
		}
		return(directions.toString());
	}

	/**
	  * Finds all of the rooms in the map that contain object.
	  *
	  * @param object a String representing the object in the MUD to be
	  *               located.
	  * @return a Vector of all the rooms in the map that contain the
	  *         object.
	  */
	public Vector findObject(String object) {
		Vector	theRooms = null;
		int	i;
		int	length;
		Room	r        = null;

		length = this.size();
		for (i = 0; i < length; i++) {
			r = ((Room)this.elementAt(i));
			if (r.containsObject(object)) {
				if (theRooms == null) {
					theRooms = new Vector();
				}
				theRooms.addElement(r);
			}
		}
		return(theRooms);
	}

	/**
	  * @deprecated
	  */
	public void readMap() {
	}
}
