/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.util.*;
import java.io.*;
import gnu.regexp.*;

/**
  * A class that implements a general map for storing information about player
  * in the collaborative environment, and a set of methods for searching for
  * players in the map. In addition there are a set of methods for reading and
  * writing the maps to disk in both Java object format and
  * pseudo-MAAS-Neotek format.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  */
public class PlayerMap extends Vector implements Serializable {
	int	id = 0;
	String	mapFilename = "tween.players.map";

	/**
	  * Create an empty player map.
	  */
	public PlayerMap() {
		super();
	}

	/**
	  * Create a player map and add the specified player.
	  *
	  * @param player the player to be added to the map.
	  */
	public PlayerMap(MudPlayer player) {
		super();
		add(player);
	}

	/**
	  * Returns the player identifier for the next player.
	  *
	  * @param the integer identifier for the next player to be added.
	  */
	public synchronized int getId() {
		return(this.id);
	}

	/**
	  * Loads the player map from disk.
	  *
	  * @return the PlayerMap that was loaded or null if there was no
	  *         map to load.
	  */
	public static PlayerMap loadMap() {
		MudPlayer	mapAry[]      = null;	/* used by the loadMap    */
						/* method to load the     */
						/* object map from disk.  */
		int	highest = -1;
		PlayerMap	map     = null;

		try {
			File f = new File("tween.players.dat");
			if (f.exists()) {
				ObjectInputStream in = new ObjectInputStream(new FileInputStream("tween.players.dat"));
				mapAry = (MudPlayer[])in.readObject();
				in.close();
				map = new PlayerMap();
				for (int j = 0; j < mapAry.length; j++) {
					if(highest < mapAry[j].getId()) {
						highest = mapAry[j].getId();
					}
					map.fastAdd(mapAry[j]);
				}

			}
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
		catch(ClassNotFoundException cnf) {
			System.out.println(cnf.toString());
		}
		return(map);
	}

	/**
	  * Adds a player to the map and writes the map back to disk.
	  *
	  * @param player the player to add to the map.
	  */
	public synchronized void add(MudPlayer player) {
		if (this.getMudPlayer(player) == null) {
			player.setId(id);
			id++;
			super.addElement(player);
			WriteMap();
			WriteMapAsObjects();
		}
	}

	/**
	  * Adds a player to the map but does not write the maps to disk.
	  *
	  * @param player the player to add to the map.
	  */
	public synchronized void fastAdd(MudPlayer player) {
		player.setId(id);
		id++;
		super.addElement(player);
	}

	/**
	  * Given a player find the player in the map.
	  *
	  * @param p the player to locate in the map.
	  * @return the player from the map or null if the player doesn't
	  *         exist.
	  */
	public MudPlayer getMudPlayer(MudPlayer p) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;

		while ((!done) &&
		       (i < length)) {
			if (p.equals((MudPlayer)elementAt(i))) {
				player = (MudPlayer)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(player);
	}

	/**
	  * Given the id of a player in the map locate and return that player.
	  *
	  * @param id of the player to locate in the map.
	  * @return the player from the map or null if the player doesn't
	  *         exist.
	  */
	public MudPlayer getMudPlayer(int id) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;

		while ((!done) &&
		       (i < length)) {
			if (((MudPlayer)elementAt(i)).getId() == id) {
				player = (MudPlayer)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(player);
	}

	/**
	  * Locate a player with the given name.
	  *
	  * @param name the name of the player to locate.
	  * @return a player with the given name from the map or null if the
	  *         player is not present in the map.
	  */
	public MudPlayer getMudPlayer(String name) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;

		while ((!done) &&
		       (i < length)) {
			if (((MudPlayer)elementAt(i)).getName().toLowerCase().equals(name.toLowerCase())) {
				player = (MudPlayer)elementAt(i);
				done = true;
			}
			else {
				i++;
			}
		}
		return(player);
	}

	/**
	  * Determines if a player in the map exists with the given name.
	  *
	  * @param name of the player to check for.
	  * @return true if the player exists, false if they do not.
	  */
	public boolean playerExists(String name) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			if (((MudPlayer)elementAt(i)).getName().equals(name)) {
				player = (MudPlayer)elementAt(i);
				rc   = true;
				done = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Determines if a player in the map exists with the given id.
	  *
	  * @param id of the player to check for.
	  * @return true if the player exists, false if they do not.
	  */
	public boolean playerExists(int id) {
		int	i = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			player = (MudPlayer)elementAt(i);
			if (player.getId() == id) {
				rc   = true;
				done = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Determines if a MudPlayer in the map exists given another
	  * MudPlayer.
	  *
	  * @param theMudPlayer to check the map for.
	  * @return true if the player exists, false if they do not.
	  */
	public boolean playerExists(MudPlayer theMudPlayer) {
		int	i      = 0;
		int	length = this.size();
		boolean	done   = false;
		MudPlayer	player   = null;
		boolean	rc     = false;

		while ((!done) &&
		       (i < length)) {
			player = (MudPlayer)elementAt(i);
			if (theMudPlayer.equals(player)) {
				done = true;
				rc   = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  * Returns a String representation of the map.
	  *
	  * @return a String representation of the map.
	  */
	public String toString() {
		StringBuffer	sb;
		int		i;
		MudPlayer		aMudPlayer;

		sb = new StringBuffer();
		for (i = 0; i < size(); i++) {
			aMudPlayer = ((MudPlayer)elementAt(i));
			sb.append(aMudPlayer.toMap());
		}
		return(sb.toString());
	}

	/**
	  * Sets the output filename for the map
	  *
	  * @param mapFilename the filename that should be used for reading and
	  *                    writing the map.
	  */
	public void setMapFilename(String mapFilename) {
		this.mapFilename = mapFilename;
	}

	/**
	  * Gets the output filename for the map
	  *
	  * @return the filename that should be used for reading and writing 
	  *         the map.
	  */
	public String getMapFilename() {
		return(this.mapFilename);
	}

	/**
	  * Write the map out to disk as a set of Java objects.
	  */
	public void WriteMapAsObjects() {
		MudPlayer	p[];

		try {
			p = new MudPlayer[this.size()];
			this.copyInto(p);
			ObjectOutputStream	out = new ObjectOutputStream(new FileOutputStream("tween.players.dat"));
			out.writeObject(p);
			out.close();
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
	}

	/**
	  * Writes the map out in psuedo-MAAS-Neotek format.
	  */
	public void WriteMap() {
		FileOutputStream	o = null;
		OutputStreamWriter	w = null;

//System.out.print(".");
		try {
			o = new FileOutputStream(getMapFilename());
			w = new OutputStreamWriter(o);
			w.write(toString());
			w.close();
		}
		catch(IOException ie) {
			System.out.println(ie.toString());
		}
	}
}
