/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;
import java.util.*;
import os.*;

/**
  * A class that emits responses from the robot to other players in the
  * collaborative environment.
  *
  * @author Thaddeus O. Cooper
  * @version 1.6
  */
public abstract class Responder {
	String			name         = null;
	MudMap			map          = null;
	String			who          = null;
	String			what         = null;
	String			how          = null;
	String			response     = null;
	String			action       = null;
	Properties		properties   = null;
	PrintStream		out          = null;
	DataInputStream		in           = null;
	Vector			actions      = null;
	PlayerMap		players      = null;
	ParseRoomDescription	roomParser   = null;
	Commands		commands     = null;
	Connection		connection   =  null;
	String			owner        = null;
	ProcessTable		processTable = null;
	int			state        = 0;
	Vector			shared       = null;

	/**
	  * Creates an empty responder.
	  */
	public Responder() {
		super();
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name.
	  *
	  * @param name  The name of the robot.
	  */
	public Responder(String name) {
		this();
		setRobotName(name);
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, and the map to the specified MudMap.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  */
	public Responder(String name, MudMap map) {
		this(name);
		setMap(map);
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, the map to the specified MudMap, and who to the name of the
	  * character that the robot is responding to.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  *  @param who   The name of the character that the robot is 
	  *               responding to.
	  */
	public Responder(String name, MudMap map, String who) {
		this(name, map);
		setWho(who);
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, the map to the specified MudMap, who to the name of the
	  * character that the robot is responding to, and what the character
	  * said to the robot.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  *  @param who   The name of the character that the robot is 
	  *               responding to.
	  *  @param what  What the character said to the robot.
	  */
	public Responder(String name, MudMap map, String who, String what) {
		this(name, map, who);
		setWhat(what);
	}
		
	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, the map to the specified MudMap, who to the name of the
	  * character that the robot is responding to, what the character
	  * said to the robot, and the input stream that the robot can listen
	  * to the collaborative environment on.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  *  @param who   The name of the character that the robot is 
	  *               responding to.
	  *  @param what  What the character said to the robot.
	  *  @param in    The input stream that the robot can listen to the
          *               environment on.
	  */
	public Responder(String name, MudMap map, String who, String what, 
		       DataInputStream in) {
		this(name, map, who, what);
		setInputStream(in);
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, the map to the specified MudMap, who to the name of the
	  * character that the robot is responding to, what the character
	  * said to the robot, the input stream that the robot can listen
	  * to the collaborative environment on, and the output stream that the
	  * robot can write to the collaborative environment on.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  *  @param who   The name of the character that the robot is 
	  *               responding to.
	  *  @param what  What the character said to the robot.
	  *  @param in    The input stream that the robot can listen to the
          *               environment on.
	  *  @param out   The output stream that the robot can output to.
	  */
	public Responder(String name, MudMap map, String who, String what,
		       DataInputStream in, PrintStream out) {
		this(name, map, who, what, in);
		setOutputStream(out);
	}

	/**
	  * Creates a responder and sets the name of the robot to the specified
	  * name, the map to the specified MudMap, who to the name of the
	  * character that the robot is responding to, what the character
	  * said to the robot, the input stream that the robot can listen
	  * to the collaborative environment on, the output stream that the
	  * robot can write to the collaborative environment on, and the
	  * properties that were set for the robot when it started.
	  *
	  *  @param name  The name of the robot.
	  *  @param map   The map of the mud that the robot has constructed.
	  *  @param who   The name of the character that the robot is 
	  *               responding to.
	  *  @param what  What the character said to the robot.
	  *  @param in    The input stream that the robot can listen to the
          *               environment on.
	  *  @param out   The output stream that the robot can output to.
	  *  @param properties The properties that the robot was started with.
	  */
	public Responder(String name, MudMap map, String who, String what,
		       DataInputStream in, PrintStream out,
		       Properties properties) {
		this(name, map, who, what, in, out);
		setProperties(properties);
	}

	/**
	  * Sets the name of the robot.
	  *
	  * @param name The name of the robot.
	  */
	public void setRobotName(String name) {
		this.name = name;
	}

	/**
	  *  Gets the name of the robot.
	  *
	  * @return The name of the robot.
	  */
	public String getRobotName() {
		return(this.name);
	}

	/**
	  * Sets the map of the mud.
	  *
	  * @param map The map of the mud that the robot has constructed.
	  */
	public void setMap(MudMap map) {
		this.map = map;
	}

	/**
	  * Gets the map of the mud.
	  *
	  * @return The map of the mud that the robot has constructed.
	  */
	public MudMap getMap() {
		return(this.map);
	}

	/**
	  * Sets who is talking to the robot.
	  *
	  * @param who The name of the character that the robot is responding 
	  *            to.
	  */
	public void setWho(String who) {
		this.who = who;
	}

	/**
	  * Gets who is talking to the robot.
	  *
	  * @return The name of the character that the robot is responding to.
	  */
	public String getWho() {
		return(this.who);
	}

	/**
	  * Sets what the person said to the robot.
	  *
	  * @param what  What the character said to the robot.
	  */
	public void setWhat(String what) {
		this.what = what;
	}

	/**
	  * Gets what the person said to the robot.
	  *
	  * @return What the character said to the robot.
	  */
	public String getWhat() {
		return(this.what);
	}

	/**
	  *  Sets how the person interacted with the robot. For example, did
	  *  they say it, page it, whisper to it.
	  *
	  * @param how they interacted with the robot.
	  */
	public void setHow(String how) {
		this.how = how;
	}

	/**
	  *  Gets how the person interacted with the robot. For example, did
	  *  they say it, page it, whisper to it.
	  *
	  @return how they interacted with the robot.
	  */
	public String getHow() {
		return(this.how);
	}

	/**
	  * Sets the robot's response.
	  *
	  * @param response the response from the robot.
	  */
	protected void setResponse(String response) {
		this.response = response;
	}

	/**
	  * Gets the robot's response.
	  *
	  * @return the response from the robot.
	  */
	public String getResponse() {
		return(this.response);
	}

	/**
	  * Sets the robot's action.
	  *
	  * @param action the robot's action.
	  */
	public void setAction(String action) {
		this.action = action;
	}

	/**
	  * Gets the robot's action.
	  *
	  * @return the robot's action.
	  */
	public String getAction() {
		return(this.action);
	}

	/**
	  * Sets the properties from when the robot was started.
	  *
	  * @param properties the robot's properties from when it was started.
	  */
	public void setProperties(Properties properties) {
		this.properties = properties;
	}

	/**
	  * Gets the properties from when the robot was started.
	  *
	  * @return the robot's properties from when it was started.
	  */
	public Properties getProperties() {
		return(this.properties);
	}

	/**
	  * Sets the input stream to listen to the collaborative environment
	  * on.
	  *
	  * @param in the stream to listen to the collaborative environment on.
	  */
	public void setInputStream(DataInputStream in) {
		this.in = in;
	}

	/**
	  * Gets the input stream to listen to the collaborative environment
	  * on.
	  *
	  * @return the stream to listen to the collaborative environment on.
	  */
	public DataInputStream getInputStream() {
		return(this.in);
	}

	/**
	  * Sets the output stream to write to the collaborative environment
	  * on.
	  *
	  * @param out the stream to write to the collaborative environment on.
	  */
	public void setOutputStream(PrintStream out) {
		this.out = out;
	}

	/**
	  * Gets the output stream to write to the collaborative environment
	  * on.
	  *
	  * @return the stream to write to the collaborative environment on.
	  */
	public PrintStream getOutputStream() {
		return(this.out);
	}

	/**
	  * Sets the current players map to the specified map.
	  *
	  * @param players the players map.
	  */
	public void setPlayerMap(PlayerMap players) {
		this.players = players;
	}

	/**
	  * Gets the current players map to the specified map.
	  *
	  * @return the players map.
	  */
	public PlayerMap getPlayerMap() {
		return(this.players);
	}

	/**
	  * Sets the room parser.
	  *
	  * @param roomParser the parser for parsing room descriptions.
	  */
	public void setRoomParser(ParseRoomDescription roomParser) {
		this.roomParser = roomParser;
	}

	/**
	  * Sets the room parser.
	  *
	  * @return the parser for parsing room descriptions.
	  */
	public ParseRoomDescription getRoomParser() {
		return(this.roomParser);
	}

	/**
	  * Sets the connection for this collaborative environment.
	  *
	  * @param connection the connection for this collaborative 
	  *                   environment.
	  */
	public void setConnection(Connection connection) {
		this.connection = connection;
	}

	/**
	  * Gets the connection for this collaborative environment.
	  *
	  * @return the connection for this collaborative environment.
	  */
	public Connection getConnection() {
		return(this.connection);
	}

	/**
	  * Sets the commands for this collaborative environment.
	  *
	  * @param commands the commands for this collaborative environment.
	  */
	public void setCommands(Commands commands) {
		this.commands = commands;
	}

	/**
	  * Gets the commands for this collaborative environment.
	  *
	  * @return the commands for this collaborative environment.
	  */
	public Commands getCommands() {
		return(this.commands);
	}

	/**
	  * Sets the name of the owner of this robot.
	  *
	  * @param owner the name of the owner of this robot.
	  */
	public void setOwner(String owner) {
		this.owner = owner;
	}

	/**
	  * Gets the name of the owner of this robot.
	  *
	  * @return the name of the owner of this robot.
	  */
	public String getOwner() {
		return(this.owner);
	}

	/**
	  * @deprecated
	  */
	public void setProcessTable(ProcessTable processTable) {
		this.processTable = processTable;
	}

	/**
	  * @deprecated
	  */
	public ProcessTable getProcessTable() {
		return(this.processTable);
	}

	/**
	  * Adds an action, pose, emote, etc. to the actions vector.
	  *
	  * @param action the type of action to perform.
	  * @param response the action to perform.
	  */
	protected void addRobotAction(String action, String response) {
		if (actions == null) {
			actions = new Vector();
		}
		actions.addElement(new RobotAction(action, response));
	}

	/**
	  * Adds a spoken response. DEFAULT indicates that the object that is
	  * using this response class should apply the appropriate type of
	  * response based on the original interacton with the robot.
	  *
	  * @param response the response that the robot should say to the
	  *                 player that it is interacting with.
	  */
	public void addResponse(String response) {
		addRobotAction("DEFAULT", response);
	}

	/**
	  * Adds an action, pose, emote, etc. to be performed.
	  *
	  * @param how the type of action to perform.
	  * @param action the action to perform.
	  */
	public void addAction(String how, String action) {
		addRobotAction(how, action);
	}

	/**
	  * Gets the list of actions that the robot is supposed to perform.
	  *
	  * @return a RobotAction array with a list of all the actions that
	  *         the robot is to perform.
	  */
	public RobotAction[] getActions() {
		RobotAction	actionAry[] = null;

		if (this.actions != null) {
			actionAry = new RobotAction[this.actions.size()];
			this.actions.copyInto(actionAry);
			this.actions.clear();
		}
		return(actionAry);
	}

	/**
	  * Sets the state of the responder. Used for stateful responses.
	  *
	  * @param state the state the responder is in.
	  */
	public void setState(int state) {
		this.state = state;
	}

	/**
	  * Gets the state of the responder. Used for stateful responses.
	  *
	  * @return the state the responder is in.
	  */
	public int getState() {
		return(this.state);
	}

	/**
	  * Sets the shared data space. The shared data space is the inter-
	  * process communication mechanism that is used by the robot for
	  * sharing information within itself.
	  *
	  * @param shared the vector of shared information.
	  */
	public void setShared(Vector shared) {
		this.shared = shared;
	}

	/**
	  * Gets the shared data space. The shared data space is the inter-
	  * process communication mechanism that is used by the robot for
	  * sharing information within itself.
	  *
	  * @return the vector of shared information.
	  */
	public Vector getShared() {
		return(this.shared);
	}

	/**
	  * Actually process the input from the collaborative environment and
	  * add the actions and responses to the Vector to be output when this
	  * routine returns.
	  */
	public abstract void process();

	/**
	  * Returns a string the defines the type of service that this 
	  * responder provides. Some examples include: NEWS, DIRECTIONS, and
	  * DEFAULT.
	  *
	  * @return the String that defines the type of service.
	  */
	public abstract String serviceType();

	/**
	  * Determines if this responder will respond to the data contained in
	  * the chatter. If the chatter is able to be processed by this
	  * responder, then it should return true, if not it should return 
	  * false.
	  *
	  * @return true if this reponder can process the input, false 
	  *         otherwise.
	  */
	public abstract boolean canProcess(Chatter c);

}
