/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.util.*;
import java.io.*;
import gnu.regexp.*;

/**
  * A class that represents a room in the collaborative environment and
  * information about that room including: name, description, database
  * reference, commands, obvious exits, possible and fallback exits, contents
  * and chatter. In addition there are methods for creating psuedo-MAAS-Neotek
  * representations of the room that can be written to map files on disk, and
  * methods for adding contents and exits.
  *
  * @author Thaddeus O. Cooper
  * @version 1.6
  */
public class Room implements Serializable {
	String			name                = null;
	String			dbReference         = null;
	String			description         = null;
	String			commands[]          = null;
	String			shownExits[]        = null;
	String			possibleExits[]     = null;
	String			contents[]          = null;
	String			chatter[]           = null;
	int			id                  = -1;
	int			shownExitsIndex     = 0;
	int			possibleExitsIndex  = 0;
	Exit			goodExits[]         = null;
	Exit			badExits[]          = null;
	Exit			suspectExits[]      = null;
	boolean			visited             = false;
	boolean			confusingPathTo     = false;
	GregorianCalendar	firstIn             = null;
	GregorianCalendar	lastIn              = null;
	long			totalTimeIn         = -1;
	boolean			usePossibleExits    = false;
	boolean			useFallbackExits    = false;
	String			fallbackExits[]     = null;
	int			fallbackExitsIndex  = 0;

	/**
	  *  This object represents a room in the Mud/MOO. It
	  *  contains a name at a minimum, and information about
	  *  the description, exits, contents, and other related
	  *  data.
	  */
	public Room() {
		super();
	}

	/**
	  *  This object represents a room in the Mud/MOO. It
	  *  contains a name at a minimum, and information about
	  *  the description, exits, contents, and other related
	  *  data.
	  *
	  *  @param name the name of this room.
	  */
	public Room(String name) {
		super();
		setName(name);
	}

	/**
	  *  This routine sets the numeric identifier for this
	  *  room. This identifier is used to uniquely identify
	  *  this room from other similar rooms.
	  *
	  *  @param id the numeric identifier for this room.
	  */
	public void setId(int id) {
		this.id = id;
	}

	/**
	  *  This routine gets the numeric identifier for this
	  *  room. This identifier is used to uniquely identify
	  *  this room from other similar rooms.
	  *
	  *  @return id the numeric identifier for this room.
	  */
	public int getId() {
		return(this.id);
	}

	/**
	  *  This routine sets the name of this room.
	  *
	  *  @param name the name of this room. This is generally derived from 
	  *              the room description returned by the Mud/MOO.
	  */
	public void setName(String name) {
		this.name = name;
	}

	/**
	  *  This routine gets the name of this room.
	  *
	  *  @return name the name of this room.
	  */
	public String getName() {
		return(this.name);
	}

	/**
	  *  This routine sets the database reference for this room.
	  *  The database reference is sometimes available from the
	  *  description that the Mud/MOO returns.
	  *
	  *  @param dbReference the database reference for this room.
	  */
	public void setDBReference(String dbReference) {
		this.dbReference = dbReference;
	}

	/**
	  *  This routine gets the database reference for this room.
	  *
	  *  @return dbReference the database reference for this room.
	  */
	public String getDBReference() {
		return(this.dbReference);
	}

	/**
	  *  This routine sets the description for this room. The
	  *  description, when available, is returned from the
	  *  description that the Mud/MOO returns.
	  *
	  *  @param description the description of this room.
	  */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	  *  This routine gets the description for this room.
	  *
	  *  @return description the description of this room or null if none 
	  *                      exists.
	  */
	public String getDescription() {
		return(this.description);
	}

	/**
	  *  This routine sets the chatter vector. The chatter 
	  *  vector stores "chatter" that was heard in the
	  *  room while waiting for expected output. Chatter is
	  *  defined as any text that the Mud/MOO generated that
	  *  was not attributed to a command that the robot executed.
	  *
	  *  @param v the chatter vector to be associated with this
	  *           room.
	  */
	public void setChatter(Vector v) {
		if ((v != null) &&
		    (v.size() > 0)) {
			this.chatter = new String[v.size()];
			v.copyInto(this.chatter);
		}
	}

	/**
	  *  This routine returns an array of stored chatter for this
	  *  room.  Chatter is defined as any text that the Mud/MOO 
	  *  generated that was not attributed to a command that the 
	  *  robot executed.
	  *
	  *  @return chatter a String array of chatter that was previously 
	  *                  heard in this room.
	  */
	public String[] getChatter() {
		return(this.chatter);
	}

	/**
	  *  This routine adds contents to the list of contents for this
	  *  room. Content items are from the list that the Mud/MOO
	  *  sent in the description of the room.
	  *
	  *  @param c the content item to be added to the list.
	  */
	public void addContents(String c) {
		String	newContents[] = null;
		int	i;

		if (this.contents == null) {
			this.contents = new String[1];
			this.contents[0] = c;
		}
		else {
			newContents = new String[this.contents.length+1];
			for (i = 0; i < this.contents.length; i++) {
				newContents[i] = this.contents[i];
			}
			newContents[i] = c;
			this.contents = newContents;
		}
	}

	/**
	  *  This routine returns a list of contents that were seen in
	  *  this room.
	  *
	  *  @return contents an array of contents that were found in this 
	  *                   room.
	  */
	public String[] getContents() {
		return(this.contents);
	}

	/**
	  *  This routine sets whether or not possible exits are used
	  *  in finding exits in the room. There are three types of
	  *  exits for any given room:<P>
	  *
	  *  <OL>
	  *     <LI><DL>
	  *            <DT>Shown Exits
	  *            <DD>These are exits that are shown in the
	  *                description of the room that is returned
	  *                by the Mud/MOO.
	  *            <P>
	  *            <DT>Possible Exits
	  *            <DD>These are exits that are typically derived
	  *                from a description of the room.
	  *            <P>
	  *            <DT>Fallback Exits
	  *            <DD>These are exits that are defined by the programmer
	  *                or user and are used when all other exits are
	  *                exhausted. These exits are especially useful when
	  *                a Mud/MOO version is being used where minimal
	  *                descriptions are available and/or when the parsing
	  *                algorithm is not tuned for the particular Mud/MOO.
	  *         </DL>
	  *  </OL>
	  *
	  * @param usePossibleExits if true use possible exits, if false do not
	  *                         use possible exits in finding exits in this
	  *                         room.
	  */
	public void setUsePossibleExits(boolean usePossibleExits) {
		this.usePossibleExits = usePossibleExits;
	}

	/**
	  *  This routine returns whether or not possible exits should be
	  *  used in finding exits in the room.
	  *
	  *  @return usePossibleExits use possible exits or not.
	  */
	public boolean getUsePossibleExits() {
		return(this.usePossibleExits);
	}

	/**
	  *  This routine sets whether or not fallback exits are used
	  *  in finding exits in the room. There are three types of
	  *  exits for any given room:<P>
	  *
	  *  <OL>
	  *     <LI><DL>
	  *            <DT>Shown Exits
	  *            <DD>These are exits that are shown in the
	  *                description of the room that is returned
	  *                by the Mud/MOO.
	  *            <P>
	  *            <DT>Possible Exits
	  *            <DD>These are exits that are typically derived
	  *                from a description of the room.
	  *            <P>
	  *            <DT>Fallback Exits
	  *            <DD>These are exits that are defined by the programmer
	  *                or user and are used when all other exits are
	  *                exhausted. These exits are especially useful when
	  *                a Mud/MOO version is being used where minimal
	  *                descriptions are available and/or when the parsing
	  *                algorithm is not tuned for the particular Mud/MOO.
	  *         </DL>
	  *  </OL>
	  *
	  * @param useFallbackExits if true use fallback exits, if false do not
	  *                         use fallback exits in finding exits in this
	  *                         room.
	  */
	public void setUseFallbackExits(boolean useFallbackExits) {
		this.useFallbackExits = useFallbackExits;
	}

	/**
	  *  This routine returns whether or not fallback exits should be
	  *  used in finding exits in the room.
	  *
	  *  @return useFallbackExits use fallback exits or not.
	  */
	public boolean getUseFallbackExits() {
		return(this.useFallbackExits);
	}

	/**
	  *  This routine sets the fallback exits.
	  *
	  *  @param fallbackExits a list of exits to try, for example a
	  *                          typical list of fallback exits might
	  *                          include:<P>
	  *
	  *                          <UL>
	  *                             <LI>north
	  *                             <LI>south
	  *                             <LI>east
	  *                             <LI>west
	  *                             <LI>n
	  *                             <LI>s
	  *                             <LI>e
	  *                             <LI>w
	  *                             <LI>forward
	  *                             <LI>backward
	  *                             <LI>up
	  *                             <LI>down
	  *                             <LI>left
	  *                             <LI>right
	  *                          </UL>
	  *
	  */
	public void setFallbackExits(String[] fallbackExits) {
		this.fallbackExits = fallbackExits;
	}

	/**
	  *  This routine returns the list of fallback exits.
	  *
	  *  @return fallbackExits the list of fallback exits.
	  */
	public String[] getFallbackExits() {
		return(this.fallbackExits);
	}

	/**
	  *  This routine determines if this room has any unexplored
	  *  exits.
	  *
	  *  @return true there are unexplored exits in this room, if false 
	  *          there are no more unexplored exits in this room.
	  */
	public boolean hasUnexploredExits() {
		boolean	rc = false;

		if (!this.confusingPathTo) {
			if (this.shownExits != null) {
				if (this.shownExitsIndex < this.shownExits.length) {
					rc = true;
				}
			}

			if ((getUsePossibleExits()) &&
			    (this.possibleExits != null)) {
				if (this.possibleExitsIndex < this.possibleExits.length) {
					rc = true;
				}
			}

			if ((getUseFallbackExits()) &&
			    (this.fallbackExits != null)) {
				if (this.fallbackExitsIndex < this.fallbackExits.length) {
					rc = true;
				}
			}
		}
		return(rc);
	}
	
	/**
	  *  This routine returns the next unexplored exit for this room
	  *  if one exits.
	  *
	  *  @return String the name of the next unexplored exit in this
	  *                 room if null this room has no, or no more, 
	  *                 unexplored exits.
	  */
	public String getNextUnexploredExit() {
		String	rc = null;

		if (hasUnexploredExits()) {
			if ((shownExits != null) &&
			    (shownExitsIndex < shownExits.length)) {
				rc = shownExits[shownExitsIndex++];
			}
			else if ((possibleExits != null) &&
				 (getUsePossibleExits()) &&
				(possibleExitsIndex < possibleExits.length)) {
				rc = possibleExits[possibleExitsIndex++];
			}
			else if ((fallbackExits != null) &&
				 (getUseFallbackExits()) &&
				(fallbackExitsIndex < fallbackExits.length)) {
				rc = fallbackExits[fallbackExitsIndex++];
			}
		}
		return(rc);
	}

	/**
	  *  @deprecated
	  */
	public boolean hasUnvisitedExits() {
		boolean	rc   = false;
		boolean done = false;
		int	i    = 0;
	
		if (goodExits != null) {
			while ((!done) && (i < goodExits.length)) {
				if (!goodExits[i].hasBeenVisited()) {
					done = true;
					rc   = true;
				}
				else {
					i++;
				}
			}
		}
		return(rc);
	}

	/**
	  *  @deprecated
	  */
	public Exit getNextUnvisitedExit() {
		Exit	rc   = null;
		boolean	done = false;
		int	i    = 0;

		if (hasUnvisitedExits()) {
			while ((!done) && (i < goodExits.length)) {
				if (!goodExits[i].hasBeenVisited()) {
					done = true;
					rc   = goodExits[i];
				}
				else {
					i++;
				}
			}
		}
		return(rc);
	}

	/**
	  *  This routine returns an exit that points to a
	  *  specific room.
	  *
	  *  @param e the identifier of the room that
	  *              the exit should point to.
	  *  @return an exit object that points to the specific room, or null
	  *          this room does not have an exit that points to the 
	  *          specific room.
	  */
	public Exit getExit(int e) {
		Exit	rc = null;
		int	i  = 0;
		boolean	done = false;

		while ((!done) && (i < goodExits.length)) {
			if (goodExits[i].getToRoom() == e) {
				rc   = goodExits[i];
				done = true;
			}
			else {
				i++;
			}
		}
		return(rc);
	}

	/**
	  *  This routine determines if this room has an
	  *  exit that points to a specific room.
	  *
	  *  @param e the identifier of the room that the
	  *              exit should point to.
	  *  @return true there is an exit in this room that points to the 
	  *          specific room, false this is not an exit in this room that 
	  *          points the the specific room.
	  */
	public boolean exitExists(int e) {
		boolean	rc = false;
		int	i  = 0;
		boolean	done = false;

		if (goodExits != null) {
			while ((!done) && (i < goodExits.length)) {
				if (goodExits[i].getToRoom() == e) {
					rc   = true;
					done = true;
				}
				else {
					i++;
				}
			}
		}
		return(rc);
	}

	/**
	  *  This routine sets the shown exits for this room.
	  *  There are three types of exits for any given room:<P>
	  *
	  *  <OL>
	  *     <LI><DL>
	  *            <DT>Shown Exits
	  *            <DD>These are exits that are shown in the
	  *                description of the room that is returned
	  *                by the Mud/MOO.
	  *            <P>
	  *            <DT>Possible Exits
	  *            <DD>These are exits that are typically derived
	  *                from a description of the room.
	  *            <P>
	  *            <DT>Fallback Exits
	  *            <DD>These are exits that are defined by the programmer
	  *                or user and are used when all other exits are
	  *                exhausted. These exits are especially useful when
	  *                a Mud/MOO version is being used where minimal
	  *                descriptions are available and/or when the parsing
	  *                algorithm is not tuned for the particular Mud/MOO.
	  *         </DL>
	  *  </OL>
	  *
	  *  @param shownExits a list of exits that were shown in the Mud/MOO 
	  *                    description for this room.
	  */
	public void setShownExits(String[] shownExits) {
		this.shownExits = shownExits;
	}

	/**
	  *  This routine returns the shown exits for this room.
	  *
	  *  @return shownExits a list of exits that were shown in the Mud/MOO 
	  *                     description for this room.
	  */
	public String[] getShownExits() {
		return(this.shownExits);
	}

	/**
	  *  This routine sets the possible exits for this room.
	  *  There are three types of exits for any given room:<P>
	  *
	  *  <OL>
	  *     <LI><DL>
	  *            <DT>Shown Exits
	  *            <DD>These are exits that are shown in the
	  *                description of the room that is returned
	  *                by the Mud/MOO.
	  *            <P>
	  *            <DT>Possible Exits
	  *            <DD>These are exits that are typically derived
	  *                from a description of the room.
	  *            <P>
	  *            <DT>Fallback Exits
	  *            <DD>These are exits that are defined by the programmer
	  *                or user and are used when all other exits are
	  *                exhausted. These exits are especially useful when
	  *                a Mud/MOO version is being used where minimal
	  *                descriptions are available and/or when the parsing
	  *                algorithm is not tuned for the particular Mud/MOO.
	  *         </DL>
	  *  </OL>
	  *
	  *  @param possibleExits a list of exits that were derived from the 
	  *                       Mud/MOO description for this room.
	  */
	public void setPossibleExits(String[] possibleExits) {
		this.possibleExits = possibleExits;
	}

	/**
	  *  This routine returns the possible exits for this room.
	  *
	  *  @return shownExits a list of exits that were derived from Mud/MOO 
	  *                     description for this room.
	  */
	public String[] getPossibleExits() {
		return(this.possibleExits);
	}

	/**
	  *  This routine sets the shown commands for this room.
	  *  The commands are derived from the description that the
	  *  Mud/MOO returns for this room.
	  *
	  *  @param commands a list of commands for this room.
	  */
	public void setCommands(String[] commands) {
		this.commands = commands;
	}

	/**
	  *  This routine returns the shown commands for this room.
	  *
	  *  @return commands a list of commands that are associated with this 
	  *                   room.
	  */
	public String[] getCommands() {
		return(this.commands);
	}

	/**
	     @deprecated
	  */
	public void setVisited(boolean visited) {
		this.visited = visited;
	}

	/**
	     @deprecated
	  */
	public boolean getVisited() {
		return(this.visited);
	}

	/**
	  *  This routine adds an exit to the good exit
	  *  list. A good exit is defined as an exit that
	  *  goes to the expected room, contained in the
	  *  toRoom field of the exit.
	  *
	  *  @param e the exit to add to the good exit list.
	  */
	public void addGoodExit(Exit e) {
		Exit	newExits[] = null;
		int	i;

		if (goodExits == null) {
			goodExits = new Exit[1];
			goodExits[0] = e;
		}
		else {
			newExits = new Exit[goodExits.length+1];
			for (i = 0; i < goodExits.length; i++) {
				newExits[i] = goodExits[i];
			}
			newExits[i] = e;
			goodExits = newExits;
		}
	}

	/**
	  *  This routine adds an exit to the bad exit
	  *  list. A bad exit is defined as an exit that
	  *  does not go to the expected room, contained 
	  *  in the toRoom field of the exit.
	  *
	  *  @param e the exit to add to the bad exit list.
	  */
	public void addBadExit(Exit e) {
		Exit	newExits[] = null;
		int	i;

		if (badExits == null) {
			badExits = new Exit[1];
			badExits[0] = e;
		}
		else {
			newExits = new Exit[badExits.length+1];
			for (i = 0; i < badExits.length; i++) {
				newExits[i] = badExits[i];
			}
			newExits[i] = e;
			badExits = newExits;
		}
	}

	/**
	  */
	public void addSuspectExit(Exit e) {
		Exit	newExits[] = null;
		int	i;

		if (suspectExits == null) {
			suspectExits = new Exit[1];
			suspectExits[0] = e;
		}
		else {
			newExits = new Exit[suspectExits.length+1];
			for (i = 0; i < goodExits.length; i++) {
				newExits[i] = suspectExits[i];
			}
			suspectExits[i] = e;
			suspectExits = newExits;
		}
	}

	/**
	  *  This routine removes an exit from the good exits
	  *  list.
	  *
	  *  @param e the exit to be removed.
	  */
	public void removeGoodExit(Exit e) {
		int	i;
		int	j          = 0;
		Exit	newExits[] = null;

		if (goodExits != null) {
			newExits = new Exit[goodExits.length - 1];
			for (i = 0; i < goodExits.length; i++) {
				if (!goodExits[i].equal(e)) {
					newExits[j++] = goodExits[i];
				}
			}
			goodExits = newExits;
		}
	}

	/**
	  *  This routine removes an exit from the bad exits
	  *  list.
	  *
	  *  @param e the exit to be removed.
	  */
	public void removeBadExit(Exit e) {
		int	i;
		int	j          = 0;
		Exit	newExits[] = null;

		if (badExits != null) {
			newExits = new Exit[badExits.length - 1];
			for (i = 0; i < badExits.length; i++) {
				if (!badExits[i].equal(e)) {
					newExits[j++] = badExits[i];
				}
			}
			badExits = newExits;
		}
	}

	/**
	  */
	public void moveGoodExitToSuspectExit(Exit e) {
		addSuspectExit(e);
		removeGoodExit(e);
	}

	/**
	  *  This routine returns the good exits that are
	  *  currently known for this room.
	  *
	  *  @return a list of known good exits for this room.
	  */
	public Exit[] getGoodExits() {
		return(goodExits);
	}

	/**
	  *  This routine returns the bad exits that are
	  *  currently known for this room.
	  *
	  *  @return a list of known bad exits for this room.
	  */
	public Exit[] getBadExits() {
		return(badExits);
	}

	/**
	  *  This routine returns the suspect exits that are
	  *  currently known for this room.
	  *
	  *  @return a list of known suspect exits for this room.
	  */
	public Exit[] getSuspectExits() {
		return(badExits);
	}

	/**
	     @deprecated
	  */
	public void clearVisitedExits() {
		Exit	good[];
		int	i;
		int	length;

		good   = getGoodExits();
		if (good != null) {
			length = good.length;
			for (i = 0; i < length; i++) {
				good[i].clearVisited();
			}
		}
	}

	/**
	  *  This routine sets whether this room has a confusing
	  *  path to it.
	  *
	  *  @param confusingPathTo true if this room has a confusing path to 
	  *                         it.
	  */
	public void setConfusingPathTo(boolean confusingPathTo) {
		this.confusingPathTo = confusingPathTo;
	}

	/**
	  *  This routine returns whether the path to this
	  *  room is confusing.
	  *
	  *  @return true if the path to this room is confusing, false 
	  *          otherwise.
	  */
	public boolean hasConfusingPathTo() {
		return(this.confusingPathTo);
	}

	/**
	  *  This routine determines if two lists of exits are
	  *  equal.
	  *
	  *  @param e1 a list of exits to be checked.
	  *  @param e2 a second list of exits to be checked.
	  *
	  *  @return true if e1 and e2 are the same set of exits, false 
	  *          otherwise.
	  */
	boolean exitsAreEqual(String[] e1, String[] e2) {
		boolean	rc = true;
		int	i;
		int	length;

		// check for the trivial case...
		if (e1.length != e2.length) {
			rc = false;
		}
		else {
			length = e1.length;
			for (i = 0; i < length; i++) {
				if (!e1[i].equals(e2[i])) {
					rc = false;
				}
			}
		}
		return(rc);
	}

	/**
	  *  This routine determines if the room r is the same
	  *  as this room. Two rooms are defined to be equal if
	  *  and only if:<P>
	  *
	  *  <UL>
	  *     <LI>The two names are equal or both null and
	  *     <LI>The two descriptions are equal or both null and
	  *     <LI>The shown exits are equal or both null
	  *  </UL>
	  *
	  *  @param r the room to be checked.
	  *  @return true if the rooms are equal, false otherwise.
	  */
	public boolean equals(Room r) {
		boolean	d = false;
		boolean	e = false;
		boolean	n = false;
	
		if ((this.getName() != null) &&
		    (r.getName() != null)) {
			n = this.getName().equals(r.getName());
		}
		else {
			if ((this.getName() == null) &&
			    (r.getName() == null)) {
				n = true;
			}
		}

		if ((this.getDescription() != null) &&
		    (r.getDescription() != null)) {
			d = this.getDescription().equals(r.getDescription());
		}
		else {
			if ((this.getDescription() == null) &&
			    (r.getDescription() == null)) {
				d = true;
			}
		}

		if ((this.shownExits != null) &&
		    (r.shownExits != null)) {
			e = exitsAreEqual(this.shownExits, r.shownExits);
		}
		else {
			if ((this.shownExits == null) &&
			    (r.shownExits == null)) {
				e = true;
			}
		}
		return(n && d && e);
	}

	/**
	  *  This routine returns the String representation for this room.
	  *
	  *  @return the name of the room.
	  */
	public String toString() {
		return(this.getName());
	}

	/**
	  *  This routine returns the String that represents this room
	  *  as a Mass NEOTEK room object.
	  *
	  *  @return the string that represents this room in a MAAS-Neotek
	  *          map file.
	  */
	public String toMap() {
		StringBuffer	sb;
		Exit		good[];
		Exit		bad[];

		good = getGoodExits();
		bad  = getBadExits();
		sb = new StringBuffer();
		if (hasConfusingPathTo()) {
			sb.append("# confusing path to this room\r\n");
		}
		sb.append("R:").append(getId()).append(" ").append(getName()).append("\r\n");
		if ((getDescription() != null) &&
		    (!getDescription().equals(""))) {
			sb.append("D:").append(getDescription()).append("\r\n");
		}
		if ((getLastIn() != null) && (getFirstIn() != null)) {
			sb.append("F:-1\t").append(getFirstIn().getTime().getTime()).append("\t").append(getLastIn().getTime().getTime()).append("\r\n");
		}
		if (good != null) {
			for (int i = 0; i < good.length; i++) {
				if (good[i].isConfusing()) {
					sb.append("# next exit confusing\r\n");
				}
				sb.append("E:").append(good[i].getToRoom()).append("\t").append(good[i].getDirection()).append("\r\n");
			}
		}
		if (bad != null) {
			for (int i = 0; i < bad.length; i++) {
				if (bad[i] != null) {
					sb.append("B:").append(bad[i].getDirection()).append("\r\n");
				}
			}
		}
		if (getContents() != null) {
			for (int i = 0; i < contents.length; i++) {
				if (contents[i] != null) {
					sb.append("C:").append(contents[i]).append("\r\n");
				}
			}
		}
		sb.append("\r\n");
		return(sb.toString());
	}

	/**
	  *  This routine sets the time that this room was first entered.
	  *
	  *  @param firstIn the time this room was first entered.
	  */
	public void setFirstIn(GregorianCalendar firstIn) {
		this.firstIn = firstIn;
	}

	/**
	  *  This routine gets the time that this room was first entered.
	  *
	  *  @return the time this room was first entered.
	  */
	public GregorianCalendar getFirstIn() {
		return(this.firstIn);
	}

	/**
	  *  This routine sets the last time that this room was entered.
	  *
	  *  @param lastIn the time this room was last entered.
	  */
	public void setLastIn(GregorianCalendar lastIn) {
		this.lastIn = lastIn;
	}

	/**
	  *  This routine gets the last time that this room was entered.
	  *
	  *  @return the last time this room was entered.
	  */
	public GregorianCalendar getLastIn() {
		return(this.lastIn);
	}

	/**
	  *  This routine sets the total time in this room.
	  *
	  *  @param totalTimeIn the total time that the robot
	  *                        was in this room.
	  */
	public void setTotalTimeIn(long totalTimeIn) {
		this.totalTimeIn = totalTimeIn;
	}

	/**
	  *  This routine gets the total time in this room.
	  *
	  *  @return the total time spent in this room.
	  */
	public long getTotalTimeIn() {
		return(this.totalTimeIn);
	}

	/**
	  *  This routine returns the shown exits for this room
	  *  as a string.
	  *
	  *  @return the shown exits for this room.
	  */
	public String shownExitsToString() {
		StringBuffer	sb = null;
		int		i;

		sb = new StringBuffer();
		if (shownExits != null) {
			for (i = 0; i < shownExits.length; i++) {
				sb.append(shownExits[i]).append(" ");
			}
		}
		return(sb.toString());
	}

	/**
	  *  This routine determines if an object is contained in this
	  *  room. The object name is compared to each item on the contains
	  *  list using the following regular expression: *object*.
	  *
	  *  @return true if the object is contained in the contents list
	  *           for this room, false otherwise.
	  */
	public boolean containsObject(String object) {
		int	i    = 0;
		boolean	done = false;
		boolean	rc   = false;
		RE	re   = null;

		if (contents != null) {
			try {
				re = new RE(".*"+object.toLowerCase()+".*");
				while ((!done) &&
				       (i < contents.length)) {
					if (re.isMatch(contents[i].toLowerCase())) {
						done = true;
						rc   = true;
					}
					else {
						i++;
					}
				}
			}
			catch(REException ree) {
			}
		} return(rc);
	}
}
