/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package os;

import java.io.*;
import java.util.*;
import util.*;

/**
  * Implements a round-robin scheduler for the Bot Operating System. Each
  * process is allowed to run as long as it wants until it throws an
  * OSProcessDone, OSProcessExec, OSProcessLoadProcess, 
  * OSProcessRelinquishControl, or OSProcessWaitForProcess exception. At that
  * point the process is removed from the run queue, and for all but
  * OSProcessDone, and OSProcessExec, put back at the bottom of the run queue
  * to run again later. When OSProcessDone is thrown the processes done
  * method is called.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.1
  */
public class OSScheduler {
	ProcessTable	processTable    = null;
	OSProcess	processToRun    = null;
	int		processId       = 0;
	String		initPath        = null;
	String		initProcessName = null;
	Shared		kvPairs         = null;
	OSProcessLoader	loader       = null;

	/**
	  * Creates an empty scheduler with no initialization path or
	  * initialization process name.
	  */
	public OSScheduler() {
		super();
		init(null, null);
	}

	/**
	  * Creates a scheduler with the specified initialization path.
	  *
	  * @param initPath
	  */
	public OSScheduler(String initPath) {
		super();
		init(initPath, null);
	}

	/**
	  * Creates a scheduler with the specified initialization path and
	  * initialization process name.
	  *
	  * @param initPath
	  * @param initProcessName
	  */
	public OSScheduler(String initPath, String initProcessName) {
		super();
		init(initPath, initProcessName);
	}

	/**
	  * This method actually does the initialization work for setting up
	  * the scheduler including: creating a process table, setting up
	  * the shared interprocess communication space, and setting the
	  * initialization path and process name to be used when the start
	  * method is called.
	  *
	  * @param initPath the path where the scheduler can locate the
	  *                 initialization process.
	  * @param initProcessName the name of the initialization process to be
	  *                        loaded.
	  */
	void init(String initPath, String initProcessName) {
		processTable         = new ProcessTable();
		this.initPath        = initPath;
		this.initProcessName = initProcessName;
		this.kvPairs         = new Shared();
	}

	/**
	  * Sets the initialization path that is used for locating the
	  * initialization process.
	  *
	  * @param initPath the initialization path where the initialization
	  *                 process lives.
	  */
	public void setInitPath(String initPath) {
		this.initPath = initPath;
	}

	/**
	  * Gets the initialization path that is used for locating the
	  * initialization process.
	  *
	  * @return the initialization path where the initialization process
	  *         lives.
	  */
	public String getInitPath() {
		return(this.initPath);
	}

	/**
	  * Sets the initialization process name that is used to start up the
	  * operating system.
	  *
	  * @param initProcessName the name of the process that is used to
	  *                        start up the operating system.
	  */
	public void setInitProcessName(String initProcessName) {
		this.initProcessName = initProcessName;
	}

	/**
	  * Gets the initialization process name that is used to start up the
	  * operating system.
	  *
	  * @return the name of the process that is used to start up the 
	  *         operating system.
	  */
	public String getInitProcessName() {
		return(this.initProcessName);
	}

	/**
	  * Starts the operating system up and begins to run processes. A
	  * process can either be a Java class that is subclassed from 
	  * OSProcess or it can be a script. A script is a text file that
	  * contains commands that are executed by an interpreter that is
	  * specified as the first line of the file. Script files must end
	  * with the .scr extention to be recognized.
	  */
	public void start() {
		OSProcess		initProcess  = null;
		boolean			done         = false;
		Class			c            = null;
		Object			o            = null;

		if ((initPath != null) && (initProcessName != null)) {
			try {
				loader = new OSProcessLoader(initPath);
				c = loader.loadClass(initProcessName, true);
				o = c.newInstance();
				initProcess = (OSProcess)o;
			}
			catch(ClassNotFoundException cnfe) {
				System.out.println(cnfe.toString());
				System.exit(-1);
			}
			catch(InstantiationException ie) {
				System.out.println(ie.toString());
				System.exit(-1);
			}
			catch(IllegalAccessException iae) {
				System.out.println(iae.toString());
				System.exit(-1);
			}
		}
		else {
			try {
				loader = new OSProcessLoader("./processes/");
				c = loader.loadClass("testInit", true);
				o = c.newInstance();
				initProcess = (OSProcess)o;
			}
			catch(ClassNotFoundException cnfe) {
				System.out.println(cnfe.toString());
				System.exit(-1);
			}
			catch(InstantiationException ie) {
				System.out.println(ie.toString());
				System.exit(-1);
			}
			catch(IllegalAccessException iae) {
				System.out.println(iae.toString());
				System.exit(-1);
			}
		}
		initProcess.setId(++this.processId);
		initProcess.setShared(this.kvPairs);
		processTable = new ProcessTable();
		processTable.addProcess(initProcess);
		while (!done) {
			processToRun = processTable.getNextProcess();
			if (processToRun != null) {
				try {
					if (processToRun.isInitialized()) {
						if (!processToRun.isWaitingFor()) {
							processToRun.run();
						}
						else {
							if (processTable.getProcess(processToRun.getWaitingFor()) != null) {
								// still running...
								processTable.addProcess(processToRun);
							}
							else {
								processToRun.run();
							}
						}
					}
					else {
						processToRun.initialize();
						processTable.addProcess(processToRun);
					}
				}
				catch(OSProcessRelinquishControl rprc) {
					processTable.addProcess(processToRun);
				}
				catch(OSProcessDone rpd) {
					processToRun.done();
				}
				catch(OSProcessLoadProcess rplp) {
					OSProcess	newProcess = null;

					processTable.addProcess(processToRun);
					newProcess = loadProcess(rplp.getProcessToLoad());
					newProcess.setParentId(0);
				}
				catch(OSProcessWaitForProcess rplp) {
					OSProcess	newProcess = null;

					processTable.addProcess(processToRun);
					newProcess = loadProcess(rplp.getProcessToWaitFor());
					newProcess.setParentId(processToRun.getId());
					processToRun.setWaitingFor(this.processId);
				}
				catch(OSProcessExecProcess rplp) {
					OSProcess	newProcess = null;

					newProcess = loadProcess(rplp.getProcessToExec());
					newProcess.setParentId(0);
				}
				catch(Exception e) {
					System.out.println(e.toString());
					e.printStackTrace();
				}
			}
			else {
			}
			try {
				Thread.sleep(50);
			}
			catch(InterruptedException ie) {
			}
			catch(NullPointerException npe) {
				System.out.println(npe.toString());
				npe.printStackTrace();
			}
			catch(RuntimeException e) {
				System.out.println(e.toString());
				e.printStackTrace();
			}
		}
	}

	/**
	  * Gets the name of a process from a process name that has command
	  * line arguments associated with it.
	  *
	  * @param processWithArgs the name of the process with the associated
	  *                        command line arguments.
	  * @return only the process name.
	  */
	String	getProcessName(String processWithArgs) {
		StringTokenizer	st = null;
		String		processName = null;

		st = new StringTokenizer(processWithArgs);
		if (st.hasMoreTokens()) {
			processName = st.nextToken();
		}
		if (processName.endsWith("&")) {
			processName = processName.substring(0, processName.length()-1);
		}
		return(processName);
	}

	/**
	  * Loads a process from disk with the specified process name. The
	  * process can either be a Java class that is subclassed from 
	  * OSProcess or it can be a script. A script is a text file that
	  * contains commands that are executed by an interpreter that is
	  * specified as the first line of the file. Script files must end
	  * with the .scr extention to be recognized.
	  *
	  * @param processName the name of the process to load.
	  * @return an OSProcess object to place on the run queue.
	  */
	OSProcess loadProcess(String processName) {
		OSProcess	newProcess = null;
		Class		c            = null;
		Object		o            = null;

		try {
			c = loader.loadClass(getProcessName(processName), true);
			o = c.newInstance();
			newProcess = (OSProcess)o;
			newProcess.setId(++this.processId);
			newProcess.setArgs(processName);
			newProcess.setShared(this.kvPairs);
			processTable.addProcess(newProcess);
		}
		catch(ClassNotFoundException cnfe) {
			BufferedReader	br = null;
			String		interpreter = null;
			String		firstLine   = null;

			// it's not a class -- maybe it's a script file...
			try {
				br = new BufferedReader(new FileReader("processes/"+getProcessName(processName)+".scr"));
				firstLine = br.readLine();
				if (firstLine.startsWith("#!")) {
					// it's a script file...
					interpreter = firstLine.substring(2);
					try {
						c = loader.loadClass(interpreter, true);
						o = c.newInstance();
						newProcess = (OSProcess)o;
						newProcess.setId(++this.processId);
						newProcess.setArgs(interpreter+" "+processName);
						newProcess.setShared(this.kvPairs);
						processTable.addProcess(newProcess);
					}
					catch(ClassNotFoundException cnfe1) {
						System.out.println(cnfe1.toString());
					}
					catch(InstantiationException ie1) {
						System.out.println(ie1.toString());
					}
					catch(IllegalAccessException iae1) {
						System.out.println(iae1.toString());
					}
				}
				else {
					System.out.println(getProcessName(processName)+": Command not found.");
				}
				br.close();
			}
			catch(FileNotFoundException fnf) {
				System.out.println(fnf.toString());
			}
			catch(IOException ioe) {
				System.out.println(ioe.toString());
			}
		}
		catch(InstantiationException ie) {
			System.out.println(ie.toString());
		}
		catch(IllegalAccessException iae) {
			System.out.println(iae.toString());
		}
		return(newProcess);
	}

	public static void main(String[] args) {
		OSScheduler	rs = null;

		rs = new OSScheduler("./processes/", "initialize");
		rs.start();
	}
}
