/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.net.*;
import mstar.*;

/**
  * Implements the Commands class for TinyMud's.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.1
  */
public class TinyMudCommands extends Commands {

	/**
	  * Send the look command to the collaborative environment.
	  *
	  * @param at the object in the collaborative environment to look at.
	  *           This parameter may be null.
	  */
	public void look(String at) {
		getOutputStream().println("look "+at);
		getOutputStream().flush();
	}

	/**
	  * Send the say command to the collaborative environment.
	  *
	  * @param what to say.
	  */
	public void say(String what) {
		getOutputStream().println("say "+what);
		getOutputStream().flush();
	}

	/**
	  * Send the yell command to the collaborative environment.
	  *
	  * @param what to yell.
	  */
	public void yell(String what) {
		getOutputStream().println("yell "+what);
		getOutputStream().flush();
	}

	/**
	  * Send the pose command to the collaborative environment.
	  *
	  * @param what fake action to perform.
	  */
	public void pose(String what) {
		getOutputStream().println(":"+what);
		getOutputStream().flush();
	}

	/**
	  * Sends a page to a specific character in the collaborative
	  * environment.
	  *
	  * @param who to send the page to.
	  * @param what the message is. What may be null. If it is null then
	  *             the character being paged will receive a message
	  *             similar to:
	  *                foo is looking for you in somewhere
	  */
	public void page(String who, String what) {
		getOutputStream().println("page "+who+" ="+what);
		getOutputStream().flush();
	}

	/**
	  * Sends a whispered message to a specific character in the
	  * collaborative environment.
	  *
	  * @param who to send the whisper to.
	  * @param what the message is.
	  */
	public void whisper(String who, String what) {
		getOutputStream().println("whisper "+who+" ="+what);
		getOutputStream().flush();
	}

	/**
	  * Sends a muttered message to everyone except the character in the
	  * collaborative environment.
	  *
	  * @param who not to send the message to.
	  * @param what the message is.
	  */
	public void mutter(String who, String what) {
		getOutputStream().println("mutter "+who+" ="+what);
		getOutputStream().flush();
	}

	/**
	  * Sends an emote command to the collaborative environment. An emote
	  * command is a cross between a page and a pose command and can be
	  * used to express actions to other players at a distance.
	  *
	  * @param who to send the emote to.
	  * @param what the emoted action is.
	  */
	public void emote(String who, String what) {
		getOutputStream().println("em "+who+" ="+what);
		getOutputStream().flush();
	}

	/**
	  * Send the go command to the collaborative environment. The go
	  * command is how characters navigate from room to room in the
	  * collaborative environment.
	  *
	  * @param where the direction to go.
	  */
	public void go(String where) {
		getOutputStream().println("go "+where);
		getOutputStream().flush();
	}

	/**
	  * A special case of the go command to send a player back to his or
	  * her default room in the collaborative environment. 
	  */
	public void goHome() {
		go("home");
		getOutputStream().flush();
	}

	/**
	  * Sends the command to the collaborative environment to end the
	  * current session.
	  */
	public void quit() {
		go("QUIT");
		getOutputStream().flush();
	}

	/**
	  * Sends the set command to the collaborative environment. The set
	  * command lets players change attributes of themselves and other
	  * objects in the collaborative environment.
	  *
	  * @param what to set.
	  * @param flag the value to set on the object.
	  */
	public void set(String what, String flag) {
		getOutputStream().println("@set "+what+"="+flag);
		getOutputStream().flush();
	}

	/**
	  * Sends the inventory command to the collaborative environment. This
	  * command tells the collaborative environment to list the objects
	  * that the character is carrying.
	  */
	public void inventory() {
		getOutputStream().println("inv");
		getOutputStream().flush();
	}

	/**
	  * Sends the OUTPUTPREFIX command to the collaborative environment.
	  * This command is used to tell the collaborative environment to send
	  * a particular string at the beginning of each command that is
	  * generated by the character.
	  *
	  * @param what the string to prefix the output of a command with.
	  */
	public void setOutputPrefix(String what) {
		getOutputStream().println("OUTPUTPREFIX "+what);
		getOutputStream().flush();
	}

	/**
	  * Sends the OUTPUTSUFFIX command to the collaborative environment.
	  * This command is used to tell the collaborative environment to send
	  * a particular string at the end of each command that is
	  * generated by the character.
	  *
	  * @param what the string to suffix the output of a command with.
	  */
	public void setOutputSuffix(String what) {
		getOutputStream().println("OUTPUTSUFFIX "+what);
		getOutputStream().flush();
	}

	/**
	  * Sends the describe command to the collaborative environment. The
	  * describe command is used to set descriptions on players and 
	  * objects in the collaborative environment.
	  *
	  * @param description the description to set.
	  */
	public void describe(String description) {
		getOutputStream().println("@describe "+description);
		getOutputStream().flush();
	}

	/**
	  * @deprecated
	  */
	public String getHowSays() {
		return("says \"");
	}

	/**
	  * @deprecated
	  */
	public String getHowPages() {
		return("pages: ");
	}

	/**
	  * @deprecated
	  */
	public String getHowWhispers() {
		return("whispers \"");
	}
}
