/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.net.*;
import java.util.*;
import gnu.regexp.*;
import mstar.*;
/**
  * Implements the connection class for TinyMud's.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.2
  */
public class TinyMudConnection extends Connection {
	static final boolean debug = true;

	Vector		chatterVector = null;

	/**
	  * Find the output prefix in the input stream.
	  */
	public void findOutputPrefix() {
		String	aLine = null;

		try {
			waitForData();
			aLine = getInputStream().readLine();
			while (!aLine.startsWith(getOutputPrefix())) {
				waitForData();
				aLine = getInputStream().readLine();
				addToChatterVector(aLine);
			}
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
	}
	
	/**
	  * Drain the connection until there is no more output waiting to be
	  * collected. If wait is set to true, wait for output it there is
          * no output currently waiting.
	  *
	  * @param wait if true, wait for output if none is currently
	  *             available. If false, and no data is available then
	  *             just return.
	  */
	public void drainConnection(boolean wait) {
		String		aLine = null;
		boolean		done  = false;

		try {
			if (wait) {
				waitForData();
			}
			done = getInputStream().available() > 0 ? false : true;
			while (!done) {
				aLine = getInputStream().readLine();
				addToChatterVector(aLine);
				if (debug) {
					System.out.println("draining: "+aLine);
				}
				done = getInputStream().available() > 0 ? false : true;
			}
		}
		catch(IOException ioe) {
			System.out.println(ioe.toString());
		}
	}

	/**
	  * Drain the output from the collaborative environment until the
	  * suffix that was set using setOutputSuffix is found. If wait is
	  * set to true wait until data becomes available from the
	  * collaborative environment. If wait is set to false and no data is
	  * available return immediately.
	  *
	  * @param wait if true, wait for output if none is currently 
	  *             available. If false, and no data is available then
	  *             just return.
	  */
	public void drainConnectionToSuffix(boolean wait) {
		String		aLine = null;
		boolean		done  = false;
		int	state          = 0;

		if (wait) {
			waitForData();
		}
		while (!done) {
			try {
				switch (state) {
					case  0 :
						waitForData();
						aLine = getInputStream().readLine();
System.out.println("draining: "+aLine);
						if (aLine.equals(getOutputPrefix())) {
							state = 1;
						}
						else {
							addToChatterVector(aLine);
						}
						break;
					case  1 :
						if (aLine.equals(getOutputSuffix())) {
							state = 2;
						}
						else {
							waitForData();
							aLine = getInputStream().readLine();
System.out.println("draining: "+aLine);
						}
						break;
					case  2 :
						done = true;
						break;
					default :
						break;
				}
			}
			catch(IOException ioe) {
				System.out.println(ioe.toString());
			}
		}
	}

	/**
	  * Returns the chatter vector that output from the collaborative
	  * environment is stored in for later processing.
	  *
	  * @return the Vector containing output from the collaborative
	  *         environment.
	  */
	public Vector getChatterVector() {
		return(chatterVector);
	}

	/**
	  * Clears the chatter vector that output from the collaborative
	  * environment is being stored in.
	  */
	public void clearChatterVector() {
		if (chatterVector != null) {
			chatterVector.clear();
		}
	}

	/**
	  * Adds the output from the collaborative environment to the chatter
	  * vector as a Chatter class. Each line of output from the 
	  * collaborative environment is parsed and abstracted to a higher
	  * level for later processing.
	  *
	  * @param what the output from the collaborative environment to be
	  *             added to the chatter vector.
	  */
	void addToChatterVector(String what) {
		GregorianCalendar	now        = new GregorianCalendar();
		int			how        = Chatter.UNDEFINED;
		String			who        = null;
		int			index      = -1;
		Chatter			chatter    = null;
		int			goesIndex  = -1;
		RE			botPagedTo = null;
		RE			emotes     = null;
		REMatch			emotesMatch = null;
		REMatch			botsMatch   = null;

		index = what.indexOf(" ");
		if (what.toLowerCase().substring(index+1).startsWith("says \"")) {
			if (what.endsWith("\"")) {
				what = what.substring(0, what.length()-1);
				index--;
			}
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.SAYS, what.toLowerCase().substring(index+8));
		}
		else if (what.toLowerCase().substring(index+1).startsWith("pages:")) {
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.PAGES, what.toLowerCase().substring(index+8));
		}
		else if (what.toLowerCase().substring(index+1).startsWith("whispers \"")) {
			if (what.endsWith("\"")) {
				what = what.substring(0, what.length()-1);
			}
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.WHISPERS, what.toLowerCase().substring(index+11));
		}
		else if (what.toLowerCase().substring(index+1).startsWith("mutters")) {
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.MUTTERS, what.toLowerCase().substring(index+9));
		}
		else if (what.toLowerCase().substring(index+1).startsWith("has arrived")) {
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.ARRIVES, null);
		}
		else if (what.toLowerCase().substring(index+1).startsWith("has left")) {
			//chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.LEAVES, null);
		}
		else if (what.toLowerCase().substring(index+1).startsWith("goes ")) {
			goesIndex = index + 1;
System.out.println("goes matched -- direction was: \""+what.substring(goesIndex+5)+"\"");
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.LEAVES, what.substring(goesIndex+5));
		}
		else if (what.toLowerCase().substring(index+1).startsWith("gives")) {
			chatter = new Chatter(now, what.toLowerCase().substring(0, index), Chatter.GIVES_OBJECT, what.toLowerCase().substring(index+7));
		}
		else if (what.toLowerCase().startsWith("you sense that")) {
			try {
				botPagedTo = new RE("^you sense that (\\w*) is looking for you in (.*)");
				if (botPagedTo.isMatch(what.toLowerCase())) {
					botsMatch = botPagedTo.getMatch(what.toLowerCase());
					chatter = new Chatter(now, 
							      what.toLowerCase().substring(botsMatch.getSubStartIndex(1), botsMatch.getSubEndIndex(1)),
							      Chatter.PAGED_TO, 
							      what.toLowerCase().substring(botsMatch.getSubStartIndex(2)));
				}
				else {
					emotes = new RE("^you sense that (\\w*) (.*)");
					emotesMatch = emotes.getMatch(what.toLowerCase());
					chatter = new Chatter(now, 
							      what.toLowerCase().substring(emotesMatch.getSubStartIndex(1), emotesMatch.getSubEndIndex(1)),
							      Chatter.EMOTES, 
							      what.toLowerCase().substring(emotesMatch.getSubStartIndex(2)));
				}
			}
			catch(REException ree) {
			}
		}
		else {
			if ((what != null) && (!what.equals(""))) {
				if (index != -1) {
					chatter = new Chatter(now, what.substring(0, index), Chatter.POSES, what.toLowerCase().substring(index+1));
				}
				else {
					chatter = new Chatter(now, "", Chatter.UNDEFINED, what.toLowerCase().substring(index+1));
				}
			}
		}
		if (this.chatterVector == null) {
			this.chatterVector = new Vector();
		}
		if (chatter != null) {
			this.chatterVector.addElement(chatter);
		}
	}

	/**
	  * Log the robot into the MUD/MOO. This routine should be called
	  * after connectToMud has been called.
	  *
	  * @param name the name of the robot.
	  * @param password the password of the robot.
	  */
	public void loginToMud(String name, String password) {
		getOutputStream().println("connect "+name+" "+password);
		drainConnection();
	}

	/**
	  * Log the robot off the MUD/MOO.
	  */
	public void logOffMud() {
		// probably need to close the sockets here...
	}
}
