/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.util.*;

/**
  * Implements a Responder class loader for loading responder plugins into the 
  * robot at runtime.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.0
  */
public class ResponderClassLoader extends ClassLoader {

	// types is this class loader object's private
	// cache of type names it has loaded. For each type
	// name, the Hashtable stores a reference to the
	// Class instance for that type
	private Hashtable types = new Hashtable();

	// basePath gives the path to which this class
	// loader appends "/.class" to get the
	// full path name of the class file to load
	private String basePath;

	public ResponderClassLoader(String basePath) {
		this.basePath = basePath;
	}

	public synchronized Class loadClass(String className,
					    boolean resolveIt) throws ClassNotFoundException {

		Class result;
		byte classData[];

		// Check the local cache
		result = (Class) types.get(className);
		if (result != null) {
			// Return a cached class
			return result;
		}

		// Check with the primordial class loader
		try {
			result = super.findSystemClass(className);
			// Return a system class
			return result;
		}
		catch (ClassNotFoundException e) {
		}

		// Don't attempt to load a system file except through
		// the primordial class loader
		if (className.startsWith("java.")) {
			throw new ClassNotFoundException();
		}

		// Try to load it from the basePath directory.
		classData = getTypeFromBasePath(className);
		if (classData == null) {
			System.out.println("GCL - Can't load class: "
				+ className);
			throw new ClassNotFoundException();
		}

		// Parse it
		result = defineClass(className, classData, 0, classData.length);
		if (result == null) {
			System.out.println("GCL - Class format error: " + className);
			throw new ClassFormatError();
		}

		if (resolveIt) {
			resolveClass(result);
		}

		// Add newly loaded type to private name cache
		types.put(className, result);

		// Return class from basePath directory
		return result;
	}

	private byte[] getTypeFromBasePath(String className) {

		InputStream in = null;
		String fileName = basePath + "/" + className + ".class";

		try {
			in = new FileInputStream(fileName);
		} 
		catch (FileNotFoundException e) {
			return null;
		}

		ByteArrayOutputStream out = new ByteArrayOutputStream();

		try {
			int c = in.read();
			while (c != -1) {
				out.write(c);
				c = in.read();
			}
		}
		catch (IOException e) {
			System.out.println("SCL: Error reading bytes of:" + className);
			e.printStackTrace();
			return null;
		}

		return out.toByteArray();
	}
}
