/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.util.*;
import gnu.regexp.*;
import util.*;
import os.*;
import mstar.*;
import org.place.pj.*;

/**
  * A Bot OS program to map the rooms in the mud.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @author Carsten Oertel
  * @version 2.1
  */
public class MapMud extends OSProcess {
	static final String	DEFAULT_ROBOT_PROPERTIES_FILENAME = "robot.properties";
	static final int	STANDARD_TIME_INTERVAL            = 10000;
	boolean			debug = false;
	Connection	mudConnection = null;
	Commands		commands      = null;
	ParseRoomDescription	roomParser = null;
	String		name          = null;	/* the name of the robot  */
	String		description   = null;	/* the description of the */
						/* robot.                 */
	boolean		shouldExplore = true;	/* when set to true the   */
						/* bot explores the       */
						/* Mud/MOO                */
	boolean		allowedToExplore = true;
	boolean		verbose	      = true;	/* if set to true the bot */
						/* prints a lot of        */
						/* information to stdout. */
	int		state         = 6;	/* the current state the  */
						/* bot is in. the bot is  */
						/* modeled as a DFA and   */
						/* this is used to keep   */
						/* track of what should   */
						/* happen next.           */
	StringBuffer	roomDescription   = new StringBuffer();
						/* the results of the     */
						/* "look" command.        */
	MudMap		map	      = null;	/* the map of all the     */
						/* rooms that the bot has */
						/* visited.               */
	int		biggestId     = 0;	/* the largest id that    */
						/* has been assigned to a */
						/* room.                  */
	Exit		theExit       = null;	/* an exit that the bot   */
						/* has gone through.      */
	boolean		exploreBreadthFirst = false;
						/* determines whether a   */
						/* depth first search     */
						/* (the default) or a     */
						/* breadth first search   */
						/* is performed to locate */
						/* rooms in the mud.      */
	boolean		usePossibleExits    = false;
						/* determines whether the */
						/* parsed description is  */
						/* use to find new rooms  */
						/* in the mud.            */
	boolean		useFallbackExits    = false;
						/* determines whether the */
						/* list of provided       */
						/* fallback exits are     */
						/* used to find new rooms */
						/* in the mud.            */
	String		fallbackExits[]     = null;
						/* a list of fallback     */
						/* exits used to find new */
						/* rooms in the mud.      */
	Properties	properties          = null;
	GregorianCalendar	whenCanMove = null;
	int			waitTimeInterval = STANDARD_TIME_INTERVAL;

	public void initialize() {
		String[]	args = null;
		Commandline	cl = null;
		Properties	properties = null;
		String		longOptions[] = {
							"properties",
							"waittime",
						};
		String		longFlags[] = {
						"explore",
						"verbose",
					      };
		String		fallbackExitString = null;
		FileInputStream	in                 = null;
		String		waitTimeIntervalString = null;

		args = getArgs();
		try {
			cl = new Commandline(args,
					     "",
					     "",
					     longFlags,
					     longOptions);
			// first things first, see if there is a
			// properties file that was specified
			if (cl.hasOption("properties")) {
				try {
					in = new FileInputStream(cl.getOption("properties"));
					properties = new Properties();
					properties.load(in);
				}
				catch (IOException ioe) {
					System.out.println("Properties file '"+
							   cl.getOption("properties")+
							   "' not found.");
					System.exit(-1);
				}
			}
			else {
				try {
					// first look in the local directory...
					in = new FileInputStream(System.getProperty("user.dir")+
								 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
					properties = new Properties();
					properties.load(in);
				}
				catch (IOException ioe) {
					try {
						// now try the home directory....
						in = new FileInputStream(System.getProperty("user.home")+
									 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
						properties = new Properties();
						properties.load(in);
					}
					catch (IOException ioe1) {
					}
				}
			}
			if (properties != null) {
				verbose = properties.getProperty("asc.verbose", "false").toLowerCase().equals("true") ? true : false;	
				exploreBreadthFirst = properties.getProperty("robot.explorebreadthfirst", "false").toLowerCase().equals("true") ? true : false;	
				usePossibleExits = properties.getProperty("robot.usepossibleexits", "false").toLowerCase().equals("true") ? true : false;	
				useFallbackExits = properties.getProperty("robot.usefallbackexits", "false").toLowerCase().equals("true") ? true : false;	
				fallbackExitString = properties.getProperty("robot.fallbackexits");
				if (fallbackExitString != null) {
					fallbackExits = Utility.commaListToArray(fallbackExitString);
				}
				shouldExplore = properties.getProperty("robot.shouldExplore", "false").toLowerCase().equals("true") ? true : false;	
				allowedToExplore = properties.getProperty("robot.allowedToExplore", "false").toLowerCase().equals("true") ? true : false;	
				waitTimeIntervalString = properties.getProperty("robot.waittimeinterval");
				if (waitTimeIntervalString != null) {
					try {
						waitTimeInterval = Integer.parseInt(waitTimeIntervalString);
					}
					catch(NumberFormatException nfe) {
						System.out.println(nfe.toString());
					}
				}
			}

			if (cl.hasOption("v")) {
				verbose = true;
			}
			if (cl.hasOption("verbose")) {
				verbose = true;
			}
			if (cl.hasOption("e")) {
				shouldExplore = true;
			}
			if (cl.hasOption("explore")) {
				shouldExplore = true;
			}
			if (cl.hasOption("waittime")) {
				waitTimeIntervalString = cl.getOption("waittime");
				if (waitTimeIntervalString != null) {
					try {
						waitTimeInterval = Integer.parseInt(waitTimeIntervalString);
					}
					catch(NumberFormatException nfe) {
						System.out.println(nfe.toString());
					}
				}
			}

			if (verbose) {
				if (exploreBreadthFirst) {
					System.out.println("exploreBreadthFirst: true");
				}
				else {
					System.out.println("exploreBreadthFirst: false");
				}
				if (usePossibleExits) {
					System.out.println("usepossibleexits: true");
				}
				else {
					System.out.println("usepossibleexits: false");
				}
				if (useFallbackExits) {
					System.out.println("usefallbackexits: true");
					if (fallbackExitString != null) {
						System.out.println("fallbackexits: "+fallbackExitString);
					}
				}
				else {
					System.out.println("usefallbackexits: false");
				}
				if (shouldExplore) {
					System.out.println("shouldExplore: true");
				}
				else {
					System.out.println("shouldExplore: false");
				}
				if (allowedToExplore) {
					System.out.println("allowedToExplore: true");
				}
				else {
					System.out.println("allowedToExplore: false");
				}
				System.out.println("waittime: "+waitTimeInterval);
			}

		}
		catch (InvalidCommandlineArgument ica) {
			System.out.println("invalid command line argument "+ica.toString());
		}
		if (mudConnection == null) {
			mudConnection = (Connection)requestObject("RobotConnection");
		}
		if (commands == null) {
			commands = (Commands)requestObject("RobotCommands");
		}
		if (map == null) {
			map = (MudMap)requestObject("MudMap");
			if (map == null) {
				map = new MudMap();
				registerObject("MudMap", map);
			}
			map.setUsePossibleExits(usePossibleExits);
		}
		if (roomParser == null) {
			roomParser = (ParseRoomDescription)requestObject("RoomDescriptionParser");
		}
		if (whenCanMove == null) {
			whenCanMove = (GregorianCalendar)requestObject("WhenCanMove");
		}
		if ((mudConnection != null) &&
		    (commands != null)      &&
		    (map != null)           &&
		    (roomParser != null)    &&
		    (whenCanMove != null)) {
			setInitialized(true);
		}
	}

	public void run() throws OSProcessRelinquishControl {
		Vector	chatterVector = null;

		mapARoom();
		throw new OSProcessRelinquishControl();
	}

	void mapARoom() {
		boolean	done = false;

		if (whenCanMove.before(new GregorianCalendar())) {
			switch(getState()) {
				case  6 :
					state6();
					done = true;
					break;
				case  8 :
					state8();
					done = true;
					break;
				case  9 :
					state9();
					done = true;
					break;
				case 10 :
					state10();
					done = true;
					break;
				case 11 :
					state11();
					done = true;
					break;
				case 12 :
					state12();
					done = true;
					break;
				case 13 :
					state13();
					done = true;
					break;
			}
		}
	}

	/**
	     Look at the room the bot is in. Then go to state six (6) to
	     start processing it.
	  */
	public void state6() {
		if (this.map == null) {
			this.map = new MudMap();
		}
		this.map.setCurrentRoom(lookHere());
		setState(8);
	}

	/**
	     This state determines if the room object was created, and if it
	     was what to do next. If the bot started and stopped in the same
	     place either the exit won't let the bot through, or looped back
	     to the same place. Either way mark the exit bad and go to state
	     ten (10). If the exit was good, then go to state nine (9).
	  */
	public void state8() {
		String		contents[]  = null;
		StringBuffer	description = null;
		int		i;

		if (debug) {
			System.out.println("main state 8");
		}
		if (this.map.getCurrentRoom() != null) {
			this.map.getCurrentRoom().setUsePossibleExits(getUsePossibleExits());
			this.map.getCurrentRoom().setUseFallbackExits(getUseFallbackExits());
			this.map.getCurrentRoom().setFallbackExits(getFallbackExits());
			if (debug) {
				System.out.println("main state 8 -- this.thisRoom != null");
			}
			if (this.map.getPreviousRoom() == null) {
				if (debug) {
					System.out.println("main state 8 -- this.oldRoom == null");
				}
				// first time through...
				setState(9);
			}
			else {
				if (debug) {
					System.out.println("main state 8 -- this.oldRoom != null");
					System.out.println("main state 8 -- comparing ===> "+this.map.getCurrentRoom().toString());
					System.out.println("main state 8 -- to        ===> "+this.map.getPreviousRoom().toString());
				}
				if (this.map.getCurrentRoom().equals(this.map.getPreviousRoom())) {
					if (debug) {
						System.out.println("main state 8 -- we didnt move");
					}
					// we didn't move, or the exit loops...
					this.map.getPreviousRoom().addBadExit(theExit);
					this.map.getPreviousRoom().setLastIn(this.map.getCurrentRoom().getLastIn());
					this.map.WriteMap();
					this.map.WriteMapAsObjects();
					this.map.setCurrentRoom(this.map.getPreviousRoom());
					setState(10);
				}
				else {
					setState(9);
				}
			}
		}
		else {
			System.out.println("thisRoom was null! Program exiting...");
			System.exit(-100);
		}
	}

	/**
 	     This is where we check the current map to see if we already
	     know about this room.
	  */
	public void state9() {
		GregorianCalendar	entered = null;

		if (debug) {
			System.out.println("state 9");
		}
		incrementWhenCanMove();
		if (map.roomExists(this.map.getCurrentRoom())) {
			entered = this.map.getCurrentRoom().getLastIn();
			this.map.setCurrentRoom(map.getRoom(this.map.getCurrentRoom()));
			this.map.getCurrentRoom().setLastIn(entered);
			if (theExit != null) {
				if (this.map.getPreviousRoom() != null) {
					if (debug) {
						System.out.println("State 9 -- setting '"+theExit.getDirection()+"' to go from '"+this.map.getPreviousRoom().getName()+"' to '"+this.map.getCurrentRoom().getName()+"'");
					}
					theExit.setToRoom(this.map.getCurrentRoom().getId());
					theExit.setFromRoom(this.map.getPreviousRoom().getId());
					this.map.getPreviousRoom().addGoodExit(theExit);
					this.map.WriteMap();
					this.map.WriteMapAsObjects();
				}
			}
			this.map.setPreviousRoom(this.map.getCurrentRoom());
			setState(10);
		}
		else {
			setState(11);
		}
	}

	/**
	     We have been to this room before, so process any
	     chatter that we have heard and then decide how to
	     proceed next....
	  */
	public void state10() {
		boolean pagedToAnotherRoom = false;

//System.out.println(map.toString());
		// this is the trivial implementation, need to worry
		// about getting paged to other rooms, and being told
		// to go to other rooms...
		if (debug) {
			if (this.map.getCurrentRoom().getShownExits() != null) {
				for (int i = 0; i < this.map.getCurrentRoom().getShownExits().length; i++) {
					System.out.println("state10 -- "+(this.map.getCurrentRoom().getShownExits())[i]);
				}
			}
			if (this.map.getCurrentRoom().getGoodExits() != null) {
				for (int i = 0; i < this.map.getCurrentRoom().getGoodExits().length; i++) {
					String toRoomName = null;
					toRoomName = map.getRoom((this.map.getCurrentRoom().getGoodExits())[i].getToRoom()).getName();
					System.out.println("state10 -- ===> from "+(this.map.getCurrentRoom().getName())+" "+(this.map.getCurrentRoom().getGoodExits())[i].toString()+" ("+toRoomName+")");
				}
			}
			if ((this.map.getPreviousRoom() != null) && (this.map.getPreviousRoom().getGoodExits() != null)) {
				String toRoomName = null;
				for (int i = 0; i < this.map.getPreviousRoom().getGoodExits().length; i++) {
					toRoomName = map.getRoom((this.map.getPreviousRoom().getGoodExits())[i].getToRoom()).getName();
					System.out.println("state10 -- ====> from "+(this.map.getPreviousRoom().getName())+" "+(this.map.getPreviousRoom().getGoodExits())[i].toString()+" ("+toRoomName+")");
				}
			}
		}
		// depth first exploration
		if (!exploreBreadthFirst) {
			if (this.map.getCurrentRoom().hasUnexploredExits()) {
				setState(12);
			}
			else {
				setState(13);
			}
		}
		else {
			// breadth first exploration
			setState(13);
		}
	}

	/**
	     We have not been to this room before, so add it to
	     the database and continue processing...
	  */
	public void state11() {
		if (debug) {
			System.out.println("State 11");
		}
		this.map.getCurrentRoom().setId(this.map.getId());
		if (theExit != null) {
			theExit.setToRoom(this.map.getCurrentRoom().getId());
			if (this.map.getPreviousRoom() != null) {
				theExit.setFromRoom(this.map.getPreviousRoom().getId());
				this.map.getPreviousRoom().addGoodExit(theExit);
			}
		}
		this.map.setPreviousRoom(this.map.getCurrentRoom());
		map.add(this.map.getCurrentRoom());
		setState(10);
	}

	/**
	     Pick an exit to try...
	  */
	public void state12() {
		String	exit = null;

		exit = this.map.getCurrentRoom().getNextUnexploredExit();
		if (exit != null) {
			if (debug) {
				System.out.println("State 12 -- Trying '"+exit+"'");
			}
			this.theExit = new Exit();
			this.theExit.setDirection(exit);
			commands.go(exit);
			mudConnection.drainConnectionToSuffix();
			setState(6);
		}
		else {
			setState(18);
		}
	}

	/**
	     All the exits in this room have been explored, so we 
	     need to decide how to proceed next...
	  */
	public void state13() {
		Room	room       = null;
		Room	targetRoom = null;
		boolean	done       = false;
		int	state      = 1;
		int	index      = -1;

		if (debug) {
			System.out.println("main state 13");
		}
		while (!done) {
			switch(state) {
				case  1 :
					if (debug) {
						System.out.println("State13 -- state 1");
					}
					room = lookHere();
					if (room == null) {
						// something bad happened...
						System.out.println("State13 -- a room came back null from a look -- exiting");
						System.exit(0);
					}
					else {
						if (debug) {
							System.out.println("State13 -- state 1 -- We are in: ===> "+room.toMap());
							try{Thread.sleep(1000);}catch(InterruptedException ie){}
						}
						if (map.getRoom(room) == null) {
							// we have somehow ended up
							// in a new room so add it
							// to the map...
							map.add(room);
						}
						state = 2;
					}
					break;
				case  2 :
					if (debug) {
						System.out.println("State13 -- state 2");
					}
					this.map.setCurrentRoom(map.getRoom(room));
					this.map.setPreviousRoom(null);
					if (debug) {
						System.out.println("State13 -- state 2 -- We are in: ===> "+room.toMap());
					}
					if (this.map.getCurrentRoom().hasUnexploredExits()) {
						state = 6;
					}
					else {
						state = 3;
					}
					break;
				case  3 :
					if (debug) {
						System.out.println("State13 -- state 3");
					}
					targetRoom = map.getRoomWithUnexploredExits(index);
					if (targetRoom == null) {
						// we are out of rooms...
						System.out.println("We have run out of rooms...");
						state = 6;
						//System.exit(0);
					}
					else {
						if (debug) {
							System.out.print("State13 -- we are going to try and get to:");
							System.out.println(targetRoom.getName());
						}
						index = targetRoom.getId();
						state = 4;
					}
					break;
				case  4 :
					if (debug) {
						System.out.println("State13 -- state 4");
					}
					if (targetRoom.getId() == 0) {
						moveToRoom("home", map.getRoom(0));
						this.map.setPreviousRoom(map.getRoom(0));
						if (exploreBreadthFirst) {
							this.map.setCurrentRoom(map.getRoom(0));
						}
						state = 1;
					}
					else {
						state = 5;
					}
					break;
				case  5 :
					if (debug) {
						System.out.println("State13 -- state 5");
					}
					try {
						goToRoom(this.map.getCurrentRoom().getId(), targetRoom.getId());
					}
					catch(ConfusedExitException cee) {
						moveToRoom("home", map.getRoom(0));
						this.map.setPreviousRoom(map.getRoom(0));
						if (exploreBreadthFirst) {
							this.map.setCurrentRoom(map.getRoom(0));
						}
					}
					state = 1;
					break;
				case  6 :
					if (debug) {
						System.out.println("State13 -- state 6");
					}
					done = true;
					setState(6);
					break;
				default :
					// should never get here...
					break;
			}
		}
	}

	/**
	  */
	public void setRobotName(String name) {
		this.name = name;
	}

	/**
	  */
	public String getRobotName() {
		return(this.name);
	}

	/**
	  */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	  */
	public String getDescription() {
		return(this.description);
	}

	/**
	  */
	public void setExploreBreadthFirst(boolean exploreBreadthFirst) {
		this.exploreBreadthFirst= exploreBreadthFirst;
	}

	/**
	  */
	public boolean getExploreBreadthFirst() {
		return(this.exploreBreadthFirst);
	}

	/**
	  */
	public void setUsePossibleExits(boolean usePossibleExits) {
		this.usePossibleExits = usePossibleExits;
		if (this.map != null) {
			this.map.setUsePossibleExits(usePossibleExits);
		}
	}

	/**
	  */
	public boolean getUsePossibleExits() {
		return(this.usePossibleExits);
	}

	/**
	  */
	public void setUseFallbackExits(boolean useFallbackExits) {
		this.useFallbackExits = useFallbackExits;
		if (this.map != null) {
			this.map.setUseFallbackExits(useFallbackExits);
		}
	}

	/**
	  */
	public boolean getUseFallbackExits() {
		return(this.useFallbackExits);
	}

	/**
	  */
	public void setFallbackExits(String[] fallbackExits) {
		this.fallbackExits = fallbackExits;
		if (this.map != null) {
			this.map.setFallbackExits(fallbackExits);
		}
	}

	/**
	  */
	public String[] getFallbackExits() {
		return(this.fallbackExits);
	}

	/**
	  */
	void setState(int state) {
		this.state = state;
	}

	/**
	  */
	int getState() {
		return(this.state);
	}

	/**
	  */
	public boolean shouldExplore() {
		return(this.shouldExplore);
	}

	/**
	  */
	public void setAllowedToExplore(boolean t) {
		this.allowedToExplore = t;
	}

	public boolean isAllowedToExplore() {
		return(this.allowedToExplore);
	}

	/**
	  */
	public void setVerbose(boolean t) {
		this.verbose = t;
	}

	/**
	  */
	public boolean beVerbose() {
		return(this.verbose);
	}

	/**
	  */
	public void setProperties(Properties properties) {
		this.properties = properties;
	}

	/**
	  */
	public Properties getProperties() {
		return(this.properties);
	}


	/**
	  */
	StringBuffer getRoomDescription() {
		StringBuffer	rd = null;
		String			aLine = null;

		try {
			mudConnection.waitForData();
			aLine = mudConnection.readLine();
			if (rd == null) {
				rd = new StringBuffer();
			}
			while (!aLine.startsWith(mudConnection.getOutputSuffix())) {
				rd.append(aLine+"\n");
				aLine = mudConnection.readLine();
			}
		}
		catch(IOException ioe) {
		}
		return(rd);
	}

	/**
	  */
	boolean moveToRoom(String direction, Room here) {
		StringBuffer	rd         = null;
		Room		aRoom      = null;
		boolean		rc         = true;

		mudConnection.drainConnection(false);
		commands.go(direction);
		try {
			Thread.sleep(100);
		}
		catch(InterruptedException ie) {
		}
		mudConnection.drainConnectionToSuffix();
		commands.look("");
		mudConnection.findOutputPrefix();
		rd = getRoomDescription();
		if (rd != null) {
			aRoom = roomParser.parse(rd.toString());
			aRoom.setUsePossibleExits(getUsePossibleExits());
			aRoom.setUseFallbackExits(getUseFallbackExits());
			aRoom.setFallbackExits(getFallbackExits());
			if (aRoom != null) {
				this.map.setPreviousRoom(this.map.getCurrentRoom());
				this.map.setCurrentRoom(map.getRoom(aRoom));
				if (!aRoom.equals(here)) {
//System.out.println("Should be in: ===> "+here.getName()+"("+here.getId()+")\n"+here.getDescription()+here.shownExitsToString());
//System.out.println("Actually are in: ===> "+aRoom.getName()+"("+aRoom.getId()+")\n"+aRoom.getDescription()+aRoom.shownExitsToString());
					// we are not where we were supposed to be...
					rc = false;
				}
			}
			else {
//System.out.println("[1] aRoom was null");
rc = false;
			}
		}
		else {
//System.out.println("[1] rd was null");
rc = false;
		}
		return(rc);
	}

	/**
	  */
	boolean goToRoom(int start, int destination) throws ConfusedExitException {
		Stack		stack          = null;
		Stack		directions     = null;
		String		aLine          = null;
		int		roomId         = -1;
		String		textDirections = null;
		Exit		home           = new Exit();
		Exit		topExit        = null;
		Exit		theDirection   = null;
		boolean		rc             = true;

// Approach 2 & 2+
		directions     = map.findPathToRoom(start, destination);
		textDirections = map.makeDirections(start, directions);
//System.out.print("the directions are: "+textDirections);
try{Thread.sleep(1000);}catch(InterruptedException ie){}
		if ((directions != null) &&
		    (!directions.empty())) {
			while ((directions != null) && 
			       (!directions.empty())) {
				theDirection = ((Exit)directions.pop());
//System.out.println("[1] Trying '"+theDirection.getDirection().trim());
				if (!moveToRoom(theDirection.getDirection().trim(), 
				       map.getRoom(theDirection.getToRoom()))) {
//System.out.println("[1] confusion reigns...");
					System.out.println(textDirections);
				}
			}
		}
		else {
			rc = false;
		}
		return(rc);
	}

	/**
	  */
	String getRoom(String description) {
		StringTokenizer	st;
		String				room = null;

		st = new StringTokenizer(description, "\n");
		if (st.hasMoreTokens()) {
			room = st.nextToken();
		}
		return(room);
	}

	public Room lookHere() {
		StringBuffer	rd = null;
		Room		room = null;
		GregorianCalendar	now = new GregorianCalendar();

		commands.look("");
		mudConnection.findOutputPrefix();
		rd = getRoomDescription();
		if (rd != null) {
			room = roomParser.parse(rd.toString());
			room.setFirstIn(now);
			room.setLastIn(now);
			room.setUsePossibleExits(getUsePossibleExits());
			room.setUseFallbackExits(getUseFallbackExits());
			room.setFallbackExits(getFallbackExits());
		}
		return(room);
	}

	void incrementWhenCanMove() {
		whenCanMove.clear(Calendar.MILLISECOND);
		//whenCanMove.add(Calendar.MILLISECOND, STANDARD_TIME_INTERVAL);
		whenCanMove.add(Calendar.MILLISECOND, waitTimeInterval);
	}

}

class ConfusedExitException extends Exception {
	public ConfusedExitException() {
		super();
	}
	public ConfusedExitException(String s) {
		super(s);
	}
}
///////
