/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;
import java.net.*;

/**
  * Commands is an abstract base class for creating a high-level representation
  * of commands for MUD's, MOO's, MUSHes, and other collaborative environments.
  * Each command that is generated from this class is only responsible for
  * sending the command to the collaborative environment.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  */
public abstract class Commands {
	DataInputStream	in  = null;
	PrintStream	out = null;

	/**
	  * Constructs a command class with no input or output streams defined.
	  */
	public Commands() {
		super();
	}

	/**
	  * Constructs a command class and defines the input and output streams
	  * for the collaborative environment.
	  *
	  * @param in the stream to listen to the collaborative environment on.
	  * @param out the stream to write to the collaborative environment on.
	  */
	public Commands(DataInputStream in, PrintStream out) {
		super();
	}

	/**
	  * Sets the input stream to listen to the collaborative environment 
	  * on.
	  *
	  * @param in the stream to listen to the collaborative environment on.
	  */
	public void setInputStream(DataInputStream in) {
		this.in = in;
	}

	/**
	  * Gets the input stream to listen to the collaborative environment 
	  * on.
	  *
	  * @return the stream to listen to the collaborative environment on.
	  */
	public DataInputStream getInputStream() {
		return(this.in);
	}

	/**
	  * Sets the output stream to write to the collaborative environment 
	  * on.
	  *
	  * @param out the stream to write to the collaborative environment on.
	  */
	public void setOutputStream(PrintStream out) {
		this.out = out;
	}

	/**
	  * Gets the output stream to write to the collaborative environment 
	  * on.
	  *
	  * @return the stream to write to the collaborative environment on.
	  */
	public PrintStream getOutputStream() {
		return(this.out);
	}

	/**
	  * Send the look command to the collaborative environment.
	  *
	  * @param at the object in the collaborative environment to look at.
	  *           This parameter may be null.
	  */
	public abstract void look(String at);

	/**
	  * Send the say command to the collaborative environment.
	  *
	  * @param what to say.
	  */
	public abstract void say(String what);

	/**
	  * Send the yell command to the collaborative environment.
	  *
	  * @param what to yell.
	  */
	public abstract void yell(String what);

	/**
	  * Send the pose command to the collaborative environment.
	  *
	  * @param what fake action to perform.
	  */
	public abstract void pose(String what);

	/**
	  * Sends a page to a specific character in the collaborative
	  * environment.
	  *
	  * @param who to send the page to.
	  * @param what the message is. What may be null and the behaviour
	  *             of the page may vary from collaborative environment to 
	  *             collaborative environment.
	  */
	public abstract void page(String who, String what);

	/**
	  * Sends a whispered message to a specific character in the 
	  * collaborative environment.
	  *
	  * @param who to send the whisper to.
	  * @param what the message is. 
	  */
	public abstract void whisper(String who, String what);

	/**
	  * Sends a muttered message to everyone except the character in the 
	  * collaborative environment.
	  *
	  * @param who not to send the message to.
	  * @param what the message is. 
	  */
	public abstract void mutter(String who, String what);

	/**
	  * Sends an emote command to the collaborative environment. An emote
	  * is a cross between a page and a pose command and can be used to
	  * express actions to other players at a distance.
	  *
	  * @param who to send the emote to.
	  * @param what the emoted action is.
	  */
	public abstract void emote(String who, String what);

	/**
	  * Send the go command to the collaborative environment. The go
	  * command is how characters navigate from room to room in the
	  * collaborative environment.
	  *
	  * @param where the direction to go.
	  */
	public abstract void go(String where);

	/**
	  * A special case of the go command to send a player back to his or
	  * her default room in the collaborative environment. This command is
	  * usually implemented as a call to the go command in this class.
	  */
	public abstract void goHome();

	/**
	  * Sends the command to the collaborative environment to end the
	  * current session.
	  */
	public abstract void quit();

	/**
	  * Sends the set command to the collaborative environment. The set
	  * command lets players change attributes of themselves and other
	  * objects in the collaborative environment.
	  *
	  * @param what to set.
	  * @param flag the value to set on the object.
	  */
	public abstract void set(String what, String flag);

	/**
	  * Sends the inventory command to the collaborative environment. This
	  * command tells the collaborative environment to list the objects 
	  * that the character is carrying.
	  */
	public abstract void inventory();

	/**
	  * Sends the OUTPUTPREFIX command to the collaborative environment.
	  * This command is used to tell the collaborative environment to send
	  * a particular string at the beginning of each command that is
	  * generated by the character.
	  *
	  * @param what the string to prefix the output of a command with.
	  */
	public abstract void setOutputPrefix(String what);

	/**
	  * Sends the OUTPUTSUFFIX command to the collaborative environment.
	  * This command is used to tell the collaborative environment to send
	  * a particular string at the end of each command that is generated by 
	  * the character.
	  *
	  * @param what the string to suffix the output of a command with.
	  */
	public abstract void setOutputSuffix(String what);

	/**
	  * Sends the describe command to the collaborative environment. The
	  * describe command is used to set descriptions on players and
	  * objects in the collaborative environment.
	  */
	public abstract void describe(String description);

	/**
	  * @deprecated
	  */
	public abstract String getHowSays();

	/**
	  * @deprecated
	  */
	public abstract String getHowPages();

	/**
	  * @deprecated
	  */
	public abstract String getHowWhispers();
}
