/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;

/**
  * This class models exits in collaborative environments. Exits are the
  * mechanisms by which characters can navigate from one room to another.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  */
public class Exit implements Serializable {
	String	exitName  = null;
	String	direction = null;
	int	toRoom    = -1;
	boolean	visited   = false;
	int	fromRoom  = -1;
	boolean	confusing = false;

	/**
	  *  An object that is used to represent exits in a Mud/MOO.
	  *  This object keeps information about the room that it
	  *  came from and the room that it is going to. In addition
	  *  the object also can hold information about whether the
	  *  exit points to a confusing room, so that this exit can
	  *  be used cautiously in giving directions or creating
	  *  paths to rooms.
	  */
	public Exit() {
		super();
	}

	/**
	  *  This routine set the name associated with this exit.
	  *
	  *  @param exitName -- the name of the exit
	  */
	public void setExitName(String exitName) {
		this.exitName = exitName;
	}

	/**
	  *  This routine gets the name associated with this exit.
	  *
	  *  @returns exitName -- the name of the exit
	  */
	public String getExitName() {
		return(this.exitName);
	}

	/**
	  *  This routine sets the direction (e.g., out, north, etc.)
	  *  that this exit goes
	  *
	  *  @param direction -- the direction
	  */
	public void setDirection(String direction) {
		this.direction = direction;
	}

	/**
	  *  This routine gets the direction
	  *
	  *  @returns direction -- the direction
	  */
	public String getDirection() {
		return(this.direction);
	}

	/**
	  *  This routine sets the room identifier of the room
	  *  that this exit goes to.
	  *
	  *  @param toRoom -- the number of the room that this
	  *                   exit goes to.
	  */
	public void setToRoom(int toRoom) {
		this.toRoom = toRoom;
	}

	/**
	  *  This routine gets the room identifier of the room
	  *  that this exit goes to.
	  *
	  *  @returns toRoom -- the number of the room that this
	  *                     exit goes to.
	  */
	public int getToRoom() {
		return(this.toRoom);
	}

	/**
	  *  This routine sets the room identifier of the room
	  *  that this exit comes from.
	  *
	  *  @param fromRoom -- the number of the room that this
	  *                     exit comes from.
	  */
	public void setFromRoom(int fromRoom) {
		this.fromRoom = fromRoom;
	}

	/**
	  *  This routine gets the room identifier of the room
	  *  that this exit comes from.
	  *
	  *  @return fromRoom -- the number of the room that this
	  *                      exit comes from.
	  */
	public int getFromRoom() {
		return(this.fromRoom);
	}

	/**
	  *  This routine returns a string that can be used in the
	  *  Mud/MOO to "go" to the room.
	  */
	public String toString() {
		return("go "+this.direction+" to "+this.toRoom);
	}

	/**
	  *  This routine sets the visited flag to the appropriate
	  *  value.
	  *
	  *  @deprecated
	  */
	public void setVisited(boolean visited) {
		this.visited = visited;
	}

	/**
	  *  This routine sets the visited flag.
	  *
	  *  @deprecated
	  */
	public void setVisited() {
		setVisited(true);
	}

	/**
	  *  This routine clears the visited flag.
	  *
	  *  @deprecated
	  */
	public void clearVisited() {
		setVisited(false);
	}

	/**
	  *  This routine returns whether or not this room
	  *  has been previously visited.
	  *
	  *  @deprecated
	  */
	public boolean hasBeenVisited() {
		return(this.visited);
	}

	/**
	  *  This routine sets whether this is a confusing exit
	  *  or not.
	  *
	  *  @param confusing -- determines if this exit should
	  *                      be considered confusing.
	  */
	public void setConfusing(boolean confusing) {
		this.confusing = confusing;
	}

	/**
	  *  This routine sets this exit to show that it is
	  *  confusing.
	  */
	public void setConfusing() {
		setConfusing(true);
	}

	/**
	  *  This routines determines if this is a confusing
	  *  exit.
	  *
	  *  @returns confusing -- determines if this exit should
	  *                        be considered confusing.
	  */
	public boolean isConfusing() {
		return(this.confusing);
	}

	/**
	  *  This routine determines if this exit is equal to
	  *  another exit.
	  *
	  *  @param e -- the exit to be tested.
	  */
	public boolean equal(Exit e) {
		boolean	rc = true;

		if ((this.toRoom != e.toRoom) ||
		    (this.fromRoom != e.fromRoom) ||
		    (!this.direction.equals(e.direction))) {
			rc = false;
		}
		return(rc);
	}
}
