/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;
import java.util.*;

/**
  * MudObject is a base class that is used to model rooms, players and other
  * objects in the collaborative environment. If has fields for the name of
  * the object, its database reference, description, numeric identifier and
  * the date and time it was first encountered, and the date and time it was
  * last encountered.
  */
public class MudObject implements Serializable {
	String			name                = null;
	String			dbReference         = null;
	String			description         = null;
	int			id                  = -1;
	GregorianCalendar	firstEncounter      = null;
	GregorianCalendar	lastEncounter       = null;

	/**
	  * Create a MUD object with the first encounter time set.
	  */
	public MudObject() {
		super();
		setFirstEncounter(new GregorianCalendar());
	}

	/**
	  * Create a MUD object with the name of the object and the first
	  * encounter time set.
	  *
	  * @param name the name of the object.
	  */
	public MudObject(String name) {
		super();
		setName(name);
		setFirstEncounter(new GregorianCalendar());
	}

	/**
	  * Sets the numeric identifier for this object.
	  *
	  * @param id the numeric identifier for this object.
	  */
	public void setId(int id) {
		this.id = id;
	}

	/**
	  *  This routine gets the numeric identifier for this
	  *  object. This identifier is used to uniquely identify
	  *  this object from other similar object.
	  *
	  *  @returns id -- the numeric identifier for this object.
	  */
	public int getId() {
		return(this.id);
	}

	/**
	  *  This routine sets the name of this object.
	  *
	  *  @param name -- the name of this object. This is generally
	  *                 derived from the object description returned
	  *                 by the Mud/MOO.
	  */
	public void setName(String name) {
		this.name = name;
	}

	/**
	  *  This routine gets the name of this object.
	  *
	  *  @returns name -- the name of this object.
	  */
	public String getName() {
		return(this.name);
	}

	/**
	  *  This routine sets the database reference for this object.
	  *  The database reference is sometimes available from the
	  *  description that the Mud/MOO returns.
	  *
	  *  @param dbReference -- the database reference for this
	  *                        object.
	  */
	public void setDBReference(String dbReference) {
		this.dbReference = dbReference;
	}

	/**
	  *  This routine gets the database reference for this object.
	  *
	  *  @returns dbReference -- the database reference for this
	  *                          object.
	  */
	public String getDBReference() {
		return(this.dbReference);
	}

	/**
	  *  This routine sets the description for this object. The
	  *  description, when available, is returned from the
	  *  description that the Mud/MOO returns.
	  *
	  *  @param description -- the description of this object.
	  */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	  *  This routine gets the description for this object.
	  *
	  *  @returns description -- the description of this object
	  *                          or null if none exists.
	  */
	public String getDescription() {
		return(this.description);
	}

	public boolean equals(MudObject m) {
		boolean	d = false;
		boolean	e = false;
		boolean	n = false;
	
		if ((this.getName() != null) &&
		    (m.getName() != null)) {
			n = this.getName().equals(m.getName());
		}
		else {
			if ((this.getName() == null) &&
			    (m.getName() == null)) {
				n = true;
			}
		}

		if ((this.getDescription() != null) &&
		    (m.getDescription() != null)) {
			d = this.getDescription().equals(m.getDescription());
		}
		else {
			if ((this.getDescription() == null) &&
			    (m.getDescription() == null)) {
				d = true;
			}
		}

		return(n && d);
	}

	/**
	  *  This routine returns the String representation for this object.
	  *
	  *  @returns the string representation of the object.
	  */
	public String toString() {
		return(this.getName());
	}

	/**
	  *  This routine returns the String that represents this object
	  *  in MAAS-Neotek format.
	  *
	  *  @returns the string that represents this object in MAAS-Neotek
	  *           format.
	  */
	public String toMap() {
		return("");
	}

	/**
	  *  This routine sets the time that this object was first encountered.
	  *
	  *  @param firstIn -- the time this object was first encountered.
	  */
	public void setFirstEncounter(GregorianCalendar firstIn) {
		this.firstEncounter = firstEncounter;
	}

	/**
	  *  This routine gets the time that this object was first encountered.
	  *
	  *  @returns the time this object was first encountered.
	  */
	public GregorianCalendar getFirstEncounter() {
		return(this.firstEncounter);
	}

	/**
	  *  This routine sets the last time that this object was encountered.
	  *
	  *  @param lastIn -- the time this object was last encountered.
	  */
	public void setLastEncounter(GregorianCalendar lastEncounter) {
		this.lastEncounter = lastEncounter;
	}

	/**
	  *  This routine gets the last time that this object was encountered.
	  *
	  *  @returns the last time this object was encountered.
	  */
	public GregorianCalendar getLastEncounter() {
		return(this.lastEncounter);
	}
}
