/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package mstar;

import java.io.*;

/**
  * A class that represents a player in the collaborative environment and
  * information about that player including: name, description, an inventory
  * of the objects that the character is carrying, and the titles and 
  * memberships that the character has earned or been granted. In addition 
  * there are methods for creating psuedo-MAAS-Neotek representations of the
  * character that can be written to map files on disk, and methods for
  * adding titles and objects.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.6
  */
public class MudPlayer extends MudObject {
	String	titlesAndMemberships[] = null;
	String	objects[]              = null;

	/**
	  * Create a player.
	  */
	public MudPlayer() {
		super();
	}

	/**
	  * Sets the String array that contains this characters titles and
	  * memberships.
	  *
	  * @param titlesAndMemberships a String array containing each of the
	  *                             titles and memberships for this
	  *                             character.
	  */
	public void setTitlesAndMemberships(String titlesAndMemberships[]) {
		this.titlesAndMemberships = titlesAndMemberships;
	}

	/**
	  * Gets the String array that contains this characters titles and
	  * memberships.
	  *
	  * @return a String array containing each of the titles and 
	  *         memberships for this character.
	  */
	public String[] getTitlesAndMemberships() {
		return(this.titlesAndMemberships);
	}

	/**
	  * Add a title to this player.
	  *
	  * @param t the title to add.
	  */
	public void addTitle(String t) {
		String	newTitles[] = null;
		int	i;

		if (this.titlesAndMemberships == null) {
			this.titlesAndMemberships = new String[1];
			this.titlesAndMemberships[0] = t;
		}
		else {
			newTitles = new String[this.titlesAndMemberships.length+1];
			for (i = 0; i < this.titlesAndMemberships.length; i++) {
				newTitles[i] = this.titlesAndMemberships[i];
			}
			newTitles[i] = t;
			this.titlesAndMemberships = newTitles;
		}
	}

	/**
	  * Sets the list of objects that the character is carrying.
	  *
	  * @param objects a String array that represents the objects that
	  *                the character is carrying.
	  */
	public void setCarrying(String objects[]) {
		this.objects = objects;
	}

	/**
	  * Gets the list of objects that the character is carrying.
	  *
	  * @return a String array that represents the objects that the 
	  *         character is carrying.
	  */
	public String[] getCarrying() {
		return(this.objects);
	}

	/**
	  * Adds an object to the list of objects that the character is
	  * carrying. If the current list of objects is null, then a new
	  * list is created before adding the object.
	  *
	  * @param o the object to be added to the list of objects that the
	  *          the character is carrying.
	  */
	public void addObject(String o) {
		String	newObjects[] = null;
		int	i;

		if (this.objects == null) {
			this.objects = new String[1];
			this.objects[0] = o;
		}
		else {
			newObjects = new String[this.objects.length+1];
			for (i = 0; i < this.objects.length; i++) {
				newObjects[i] = this.objects[i];
			}
			newObjects[i] = o;
			this.objects = newObjects;
		}
	}

	/**
	  * Creates a String representation of the character in 
	  * psuedo-MAAS-Neotek format suitable for writing to disk.
	  *
	  * @return a String representation of the character in 
	  * pseudo-MAAS-Neotek for suitable for writing to disk.
	  */
	public String toMap() {
		StringBuffer	mapEntry = null;
		int		i;
		String		carrying[] = null;
		String		titles[]   = null;

		mapEntry = new StringBuffer();
		mapEntry.append("N:").append(getName()).append("\n");
		if (getDescription() != null) {
			mapEntry.append("D:").append(getDescription()).append("\n");
		}
		if (getTitlesAndMemberships() != null) {
			titles = getTitlesAndMemberships();
			for (i = 0; i < titles.length; i++) {
				mapEntry.append("T:").append(titles[i]).append("\n");
			}
		}
		if (getCarrying() != null) {
			carrying = getCarrying();
			for (i = 0; i < carrying.length; i++) {
				mapEntry.append("C:").append(carrying[i]).append("\n");
			}
		}
		mapEntry.append("\n");
		return(mapEntry.toString());
	}
}
