/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
package os;

import java.io.*;
import java.util.*;
import util.*;

/**
  * OSProcess is a abstract base class used to write processes that run under
  * the Bot Operating System. The process contains information about the id of
  * the process, the parent id of the process, whether or not the process is
  * waiting for another process to complete, whether it is initialized and what
  * its arguments list is. In addition it has access to the shared interprocess
  * communication space so that it can communicate with other processes that
  * are running.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.1
  */
public abstract class OSProcess {
	int		id          = -1;
	int		parentId    = -1;
	int		waitingFor  = -1;
	boolean		initialized = false;
	Object		o           = null;
	int		exitStatus  = 0;
	String[]	args        = null;
	Shared		shared      = null;

	/**
	  * Create a process that can run in the Bot Operating System.
	  */
	public OSProcess() {
		super();
		args = new String[1];
		args[0] = "";
	}

	/**
	  * Sets the id of this process.
	  *
	  * @param id the id to set this process to.
	  */
	public void setId(int id) {
		this.id = id;
	}

	/**
	  * Gets the id of this process.
	  *
	  * @return the id that this process is set to.
	  */
	public int getId() {
		return(this.id);
	}

	/**
	  * Sets the parent id of this process.
	  *
	  * @param parentId the parentId to set this process to.
	  */
	public void setParentId(int parentId) {
		this.parentId = parentId;
	}

	/**
	  * Gets the parentId of this process.
	  *
	  * @return the parentId of this process.
	  */
	public int getParentId() {
		return(this.parentId);
	}

	/**
	  * Sets the process id of the process this process is waiting for.
	  *
	  * @param waitingFor the id of the process that this process is
	  *                   waiting to complete.
	  */
	public void setWaitingFor(int waitingFor) {
		this.waitingFor = waitingFor;
	}

	/**
	  * Gets the process id of the process this process is waiting for.
	  *
	  * @return the id of the process that this process is waiting to 
	  *         complete.
	  */
	public int getWaitingFor() {
		return(this.waitingFor);
	}

	/**
	  * Determines if this process is waiting for another process to
	  * complete.
	  *
	  * @return true if this process is waiting for another process to
	  *         complete, otherwise false.
	  */
	public boolean isWaitingFor() {
		return(this.waitingFor != -1);
	}

	/**
	  * Sets the exit status for this process.
	  *
	  * @param exitStatus the exit status of this process.
	  */
	public void setExitStatus(int exitStatus) {
		this.exitStatus = exitStatus;
	}

	/**
	  * Gets the exit status for this process.
	  *
	  * @return the exit status of this process.
	  */
	public int getExitStatus() {
		return(this.exitStatus);
	}

	/**
	  * This is called by the operating system during the initialization 
	  * phase of the process. By default this method sets the process to 
	  * the initialized state and returns. More sophisticated processes 
	  * should override this method and place their initialization code 
	  * here. When initialization is complete, a call to setInitialized 
	  * with a value of true will set the process to its initialized state.
	  */
	public void initialize() {
		this.initialized = true;
	}

	/**
	  * Sets the shared interprocess communication area for this process.
	  *
	  * @param shared the shared interprocess communication area.
	  */
	public void setShared(Shared shared) {
		this.shared = shared;
	}

	/**
	  * Gets the shared interprocess communication area for this process.
	  *
	  * @return the shared interprocess communication area.
	  */
	public Shared getShared() {
		return(this.shared);
	}

	/**
	  * Register an object in the shared interprocess communication
	  * space.
	  *
	  * @param name the name of the object for reference in the shared
	  *             interprocess communication space.
	  * @param o the object to add to the shared interprocess communication
	  *          space.
	  */
	public void registerObject(String name, Object o) {
		KeyValue	kv = null;

		kv = new KeyValue(name, o);
		this.shared.addElement(kv);
	}

	/**
	  * Request an object from the shared interprocess communication space.
	  *
	  * @param name the name that the object was registered with.
	  * @return the object from the shared interprocess communication
	  *         space. It is the responsibility of the calling method to
	  *         know in advance the type of the object and cast it
	  *         appropriately.
	  */
	public Object requestObject(String name) {
		KeyValue	kv = null;
		Object		o  = null;

		kv = this.shared.findKeyValuePair(name, this.shared);
		if (kv != null) {
			o = kv.getObject();
		}
		return(o);
	}

	/**
	  * The method that is called to do the real work of the process. The
	  * process should not hold the CPU forever, or no other processes
	  * will be able to run.
	  *
	  * @exception OSProcessRelinquishControl thrown when the process wants
	  *                                       to temporarily relinquish
	  *                                       control and let another
	  *                                       process run.
	  * @exception OSProcessDone thrown when the process is finished.
	  * @exception OSProcessLoadProcess thrown when the process is 
	  *                                 requesting that the operating 
	  *                                 system load a process for it.
	  * @exception OSProcessExecProcess thrown when the process is 
	  *                                 requesting that the operating 
	  *                                 system load a process and replace 
	  *                                 it with the new process. This
	  *                                 process exits by default.
	  * @exception OSProcessWaitForProcess thrown when the process wants
	  *                                    to wait for another process to
	  *                                    complete before continuing to
	  *                                    run.
	  */
	public abstract void run() throws OSProcessRelinquishControl, 
					  OSProcessDone, 
					  OSProcessLoadProcess, 
					  OSProcessExecProcess, 
					  OSProcessWaitForProcess;

	/**
	  * This is called by the operating system during the clean up phase
	  * of the process. By default it does nothing. Processes that
	  * allocate resources should dispose of them here.
	  */
	public void done() {
	}

	/**
	  * Determines if the process has been initialized. Usually called by
	  * the operating system to determine if it should call the
	  * initialize method, or the run method.
	  *
	  * @return true if the process is initialized, false otherwise.
	  */
	public boolean isInitialized() {
		return(this.initialized);
	}

	/**
	  * Sets the processes initialized flag.
	  *
	  * @param initialized the flag that specifies whether the process has
	  *                    been initialized.
	  */
	public void setInitialized(boolean initialized) {
		this.initialized = initialized;
	}

	/**
	  * Sets an object for use by this process.
	  *
	  * @param o the object to be set.
	  */
	public void setObject(Object o) {
		this.o = o;
	}

	/**
	  * Gets an object for use by this process.
	  *
	  * @return the object.
	  */
	public Object getObject() {
		return(this.o);
	}

	/**
	  * Sets the arguments array to the specified String array.
	  *
	  * @param args the arguments to the program.
	  */
	public void setArgs(String[] args) {
		this.args = args;
	}

	/**
	  * Sets the arguments from the string that is specified. Quoted
	  * arguments are kept together as a single argument otherwise
	  * arguments are parsed on space boundaries.
	  *
	  * @param args the args to parse.
	  */
	public void setArgs(String args) {
		Vector	vector = null;
		int	i;
		int	state  = 0;
		StringBuffer	arg = null;

		i = 0;
		while (i < args.length()) {
			switch(state) {
				case  0 :
					if (args.charAt(i) == ' ') {
						state = 1;
					}
					else if (args.charAt(i) == '"') {
						if ((arg != null) && (arg.length() != 0)) {
							if (vector == null) {
								vector = new Vector();
							}
							vector.addElement(arg.toString());
							//arg.setLength(0);
							arg = new StringBuffer();
						}
						i++;
						state = 2;
					}
					else {
						if (arg == null) {
							arg = new StringBuffer();
						}
						arg.append(args.charAt(i));
						i++;
					}
					break;
				case  1 :
					if ((arg != null) && (arg.length() != 0)) {
						if (vector == null) {
							vector = new Vector();
						}
						vector.addElement(arg.toString());
						//arg.setLength(0);
						arg = new StringBuffer();
					}
					i++;
					state = 0;
					break;
				case  2 :
					if (args.charAt(i) == '"') {
						state = 3;
					}
					else {
						if (arg == null) {
							arg = new StringBuffer();
						}
						arg.append(args.charAt(i));
						i++;
					}
					break;
				case  3 :
					if ((arg != null) && (arg.length() != 0)) {
						if (vector == null) {
							vector = new Vector();
						}
						vector.addElement(arg.toString());
						//arg.setLength(0);
						arg = new StringBuffer();
					}
					i++;
					state = 0;
					break;
				default :
					// should never get here...
					break;
			}
		}
		if ((arg != null) && (arg.length() > 0)) {
			if (vector == null) {
				vector = new Vector();
			}
			vector.addElement(arg.toString());
		}
		this.args = new String[vector.size()];
		vector.copyInto(this.args);
	}

	/**
	  * Gets the arguments that were passed to this process.
	  *
	  * @return a String array of arguments that were passed to this
	  *         process.
	  */
	public String[] getArgs() {
		return(this.args);
	}

	public String toString() {
		StringBuffer	sb = null;

		sb = new StringBuffer();
		sb.append(id).append("        ").append(parentId).append("        ").append(waitingFor).append("        ").append(initialized);
		return(sb.toString());
	}

	/**
	  * Finds a key value pair.
	  *
	  * @param key the name of the value to be retrieved.
	  * @param pairs the Vector of key value pairs to be search.
	  * @return a KeyValue from the vector.
	  */
	KeyValue findKeyValuePair(String key, Vector pairs) {
		int		length;
		int		i = 0;
		boolean		done = false;
		KeyValue	rc = null;
		KeyValue	thisPair = null;

		length = pairs.size();
		while ((i < length) && (!done)) {
			thisPair = (KeyValue)pairs.elementAt(i);
			if (thisPair.getKey().toLowerCase().equalsIgnoreCase(key)) {
				done = true;
				rc = thisPair;
			}
			else {
				i++;
			}
		}
		return(rc);
	}
}
