/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.util.*;
import os.*;
import mstar.*;
import org.place.pj.*;

/**
  * A Bot OS program to load the responders for the robot.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.0
  */
public class LoadResponders extends OSProcess {
	static final String	DEFAULT_ROBOT_PROPERTIES_FILENAME="robot.properties";
	int		state = 0;
	DataInputStream	in    = null;
	PrintStream	out   = null;
	FileInputStream propertiesIn = null;
	String		robotName = null;
	boolean		customizedOnly = false;
	boolean		verbose	      = true;	/* if set to true the bot */
						/* prints a lot of        */
						/* information to stdout. */

	public void initialize() {
		String[]	args = null;
		Commandline	cl = null;
		Properties	properties = null;
		String		longOptions[] = {
							"properties",
							"name",
						};
		String		longFlags[] = {
						"customized",
						"verbose",
					      };

		args = getArgs();
		try {
			cl = new Commandline(args,
					     "nv",
					     "",
					     longFlags,
					     longOptions);
			// first things first, see if there is a
			// properties file that was specified
			if (cl.hasOption("properties")) {
				try {
					propertiesIn = new FileInputStream(cl.getOption("properties"));
					properties = new Properties();
					properties.load(propertiesIn);
				}
				catch (IOException ioe) {
					System.out.println("Properties file '"+
							   cl.getOption("properties")+
							   "' not found.");
					System.exit(-1);
				}
			}
			else {
				try {
					// first look in the local directory...
					propertiesIn = new FileInputStream(System.getProperty("user.dir")+
								 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
					properties = new Properties();
					properties.load(propertiesIn);
				}
				catch (IOException ioe) {
					try {
						// now try the home directory....
						propertiesIn = new FileInputStream(System.getProperty("user.home")+
									 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
						properties = new Properties();
						properties.load(propertiesIn);
					}
					catch (IOException ioe1) {
					}
				}
			}
			if (properties != null) {
				verbose = properties.getProperty("robot.verbose", "false").toLowerCase().equals("true") ? true : false;	
				customizedOnly = properties.getProperty("robot.customizedRespondersOnly", "false").toLowerCase().equals("true") ? true : false;	
				robotName = properties.getProperty("robot.name");
			}

			if (cl.hasOption("v")) {
				verbose = true;
			}
			if (cl.hasOption("verbose")) {
				verbose = true;
			}
			if (cl.hasOption("n")) {
				robotName = cl.getOption("n");
			}
			if (cl.hasOption("name")) {
				robotName = cl.getOption("name");
			}
			if (cl.hasOption("customized")) {
				customizedOnly = true;
			}

			if (verbose) {
				if (robotName != null) {
					System.out.println("name: "+robotName);
				}
				else {
					System.out.println("name: null");
				}
				if (customizedOnly) {
					System.out.println("customizedOnly: true");
				}
				else {
					System.out.println("customizedOnly: false");
				}
			}

		}
		catch (InvalidCommandlineArgument ica) {
			System.out.println("invalid command line argument "+ica.toString());
		}
		setInitialized(true);
	}

	Responder[] mergeResponders(Responder[] defaults, Responder[] customized) {
		Responder[]	all         = null;
		int		i           = 0;
		int		j           = 0;
		int		totalLength = 0;

		if ((defaults != null) && (customized != null)) {
			if ((customized[customized.length - 1].serviceType().equals("DEFAULT")) &&
			    (defaults[defaults.length - 1].serviceType().equals("DEFAULT"))) {
				totalLength = customized.length + defaults.length - 1;
				all = new Responder[totalLength];
				for (i = 0; i < customized.length - 1; i++) {
					all[j++] = customized[i];
				}
				for (i = 0; i < defaults.length - 1; i++) {
					all[j++] = defaults[i];
				}
				all[j] = customized[customized.length - 1];
			}
			else if (customized[customized.length - 1].serviceType().equals("DEFAULT")) {
				totalLength = customized.length + defaults.length - 1;
				all = new Responder[totalLength];
				for (i = 0; i < customized.length - 1; i++) {
					all[j++] = customized[i];
				}
				for (i = 0; i < defaults.length; i++) {
					all[j++] = defaults[i];
				}
				all[j] = customized[customized.length - 1];
			}
			else if (defaults[defaults.length - 1].serviceType().equals("DEFAULT")) {
				totalLength = customized.length + defaults.length;
				all = new Responder[totalLength];
				for (i = 0; i < customized.length; i++) {
					all[j++] = customized[i];
				}
				for (i = 0; i < defaults.length; i++) {
					all[j++] = defaults[i];
				}
			}
		}
		else if (defaults != null) {
			all = new Responder[defaults.length];
			for (i = 0; i < defaults.length; i++) {
				all[i] = defaults[i];
			}
		}
		else {
			all = new Responder[customized.length];
			for (i = 0; i < customized.length; i++) {
				all[i] = customized[i];
			}
		}
		return(all);
	}

	Responder[] loadResponders() {
		Responder[]	allResponders = null;
		Responder[]	defaultResponders = null;
		Responder[]	customizedResponders = null;

		defaultResponders    = loadDefaultResponders();
		customizedResponders = loadCustomizedResponders();
		allResponders        = mergeResponders(defaultResponders, customizedResponders);
		return(allResponders);
	}

	Responder[] loadResponders(String path) {
		File			file           = null;
		File			list[]         = null;
		int			i              = 0;
		Vector			classes        = null;
		ResponderClassLoader	loader         = null;
		Responder		responderClass = null;
		String			className      = null;
		int			sepIndex       = 0;
		int			dotIndex       = 0;
		Class			c              = null;
		Object			o              = null;
		Responder		defaultResponderClass = null;
		StringBuffer		responderPath         = null;
		Responder[]		responders            = null;

		try {
			file = new File(path);
			list = file.listFiles(new ResponderClassFilenameFilter());
			if (list != null) {
				for (i = 0; i < list.length; i++) {
					System.out.println(list[i].toString());
					if (loader == null) {
						loader = new ResponderClassLoader(path);
					}
					if (classes == null) {
						classes = new Vector();
					}
					sepIndex = list[i].toString().lastIndexOf(File.separator);
					dotIndex = list[i].toString().lastIndexOf(".");
					className = list[i].toString().substring(sepIndex + 1, dotIndex);
					System.out.println(className);
					c = loader.loadClass(className, true);
					o = c.newInstance();
					if (!className.toLowerCase().startsWith("default")) {
						responderClass = (Responder)o;
						classes.addElement(responderClass);
					}
					else {
						defaultResponderClass = (Responder)o;
					}
				}
				if (defaultResponderClass != null) {
					classes.addElement(defaultResponderClass);
				}
				if (classes != null) {
					responders = new Responder[classes.size()];
					classes.copyInto(responders);
				}
			}
		}
		catch(ClassNotFoundException cnfe) {
			System.out.println(cnfe.toString());
		}
		catch(IllegalAccessException iae) {
			System.out.println(iae.toString());
		}
		catch(InstantiationException ie) {
			System.out.println(ie.toString());
		}
		return(responders);
	}

	Responder[] loadCustomizedResponders() {
		StringBuffer		responderPath         = null;
		Responder[]		responders            = null;

		if (robotName != null) {
			responderPath = new StringBuffer();
			responderPath.append("plugins").append(File.separator).append("responders").append(File.separator).append(robotName);
			responders = loadResponders(responderPath.toString());
		}
		return(responders);
	}

	Responder[] loadDefaultResponders() {
		StringBuffer		responderPath         = null;
		Responder[]		responders            = null;

		if (!customizedOnly) {
			responderPath = new StringBuffer();
			responderPath.append("plugins").append(File.separator).append("responders");
			responders = loadResponders(responderPath.toString());
		}
		return(responders);
	}

	public void run() throws OSProcessDone {
		Responder[]	responders = null;

		responders = loadResponders();
		registerObject("RobotResponders", responders);
		throw new OSProcessDone();
	}
}
