/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.util.*;
import gnu.regexp.*;
import util.*;
import os.*;
import mstar.*;
import org.place.pj.*;

/**
  * A Bot OS program to gather information about players that the robot
  * encounters while in the collaborative environment.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.1
  */
public class LogPlayers extends OSProcess {
	static final String	DEFAULT_ROBOT_PROPERTIES_FILENAME = "robot.properties";
	static final int	STANDARD_TIME_INTERVAL            = 60000;
	boolean			debug = true;
	Connection		mudConnection = null;
	Commands		commands      = null;
	ParseRoomDescription	roomParser = null;
	ParsePlayer		playerParser = null;
	boolean		verbose	      = true;	/* if set to true the bot */
						/* prints a lot of        */
						/* information to stdout. */
	int		state         = 0;	/* the current state the  */
						/* bot is in. the bot is  */
						/* modeled as a DFA and   */
						/* this is used to keep   */
						/* track of what should   */
						/* happen next.           */
	PlayerMap	map	      = null;	/* the map of all the     */
						/* rooms that the bot has */
						/* visited.               */
	int		biggestId     = 0;	/* the largest id that    */
						/* has been assigned to a */
						/* room.                  */
	Properties		properties               = null;
	GregorianCalendar	whenShouldLook   = null;
	int			waitTimeInterval = STANDARD_TIME_INTERVAL;

	/**
	  */
	public void initialize() {
		String[]	args = null;
		Commandline	cl = null;
		Properties	properties = null;
		String		longOptions[] = {
							"properties",
							"waittime",
						};
		String		longFlags[] = {
						"verbose",
					      };
		FileInputStream	in                 = null;
		String		waitTimeIntervalString = null;

		args = getArgs();
		try {
			cl = new Commandline(args,
					     "",
					     "",
					     longFlags,
					     longOptions);
			// first things first, see if there is a
			// properties file that was specified
			if (cl.hasOption("properties")) {
				try {
					in = new FileInputStream(cl.getOption("properties"));
					properties = new Properties();
					properties.load(in);
				}
				catch (IOException ioe) {
					System.out.println("Properties file '"+
							   cl.getOption("properties")+
							   "' not found.");
					System.exit(-1);
				}
			}
			else {
				try {
					// first look in the local directory...
					in = new FileInputStream(System.getProperty("user.dir")+
								 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
					properties = new Properties();
					properties.load(in);
				}
				catch (IOException ioe) {
					try {
						// now try the home directory....
						in = new FileInputStream(System.getProperty("user.home")+
									 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
						properties = new Properties();
						properties.load(in);
					}
					catch (IOException ioe1) {
					}
				}
			}
			if (properties != null) {
				verbose = properties.getProperty("asc.verbose", "false").toLowerCase().equals("true") ? true : false;	
				waitTimeIntervalString = properties.getProperty("robot.waittimeinterval");
				if (waitTimeIntervalString != null) {
					try {
						waitTimeInterval = Integer.parseInt(waitTimeIntervalString);
					}
					catch(NumberFormatException nfe) {
						System.out.println(nfe.toString());
					}
				}
			}

			if (cl.hasOption("v")) {
				verbose = true;
			}
			if (cl.hasOption("verbose")) {
				verbose = true;
			}
			if (cl.hasOption("waittime")) {
				waitTimeIntervalString = cl.getOption("waittime");
				if (waitTimeIntervalString != null) {
					try {
						waitTimeInterval = Integer.parseInt(waitTimeIntervalString);
					}
					catch(NumberFormatException nfe) {
						System.out.println(nfe.toString());
					}
				}
			}

			if (verbose) {
				System.out.println("waittime: "+waitTimeInterval);
			}

		}
		catch (InvalidCommandlineArgument ica) {
			System.out.println("invalid command line argument "+ica.toString());
		}
		if (mudConnection == null) {
			mudConnection = (Connection)requestObject("RobotConnection");
		}
		if (commands == null) {
			commands = (Commands)requestObject("RobotCommands");
		}
		if (roomParser == null) {
			roomParser = (ParseRoomDescription)requestObject("RoomDescriptionParser");
		}
		if (playerParser == null) {
			playerParser = (ParsePlayer)requestObject("PlayerParser");
		}
		if (map == null) {
			map = (PlayerMap)requestObject("PlayerMap");
			if (map == null) {
				map = new PlayerMap();
				registerObject("PlayerMap", map);
			}
		}
		if ((mudConnection != null) && 
		    (commands != null)      && 
		    (roomParser != null)    && 
		    (playerParser != null)  && 
		    (map != null)) {
			whenShouldLook = new GregorianCalendar();
			setInitialized(true);
		}
	}

	/**
	  */
	public void run() throws OSProcessRelinquishControl {
		MudPlayer	aPlayer  = null;
		String[]	contents = null;
		Room		theRoom  = null;
		int		i;
		StringBuffer	description = null;

		if (whenShouldLook.before(new GregorianCalendar())) {
			theRoom = lookHere();
			if (theRoom != null) {
				contents = theRoom.getContents();
				if (contents != null) {
					for (i = 0; i < contents.length; i++) {
						commands.look(contents[i]);
						mudConnection.findOutputPrefix();
						description = getObjectDescription();
						aPlayer = playerParser.parse(description.toString());
						if (aPlayer != null) {
							if ((aPlayer.getCarrying() != null) ||
							    (aPlayer.getTitlesAndMemberships() != null)) {
								aPlayer.setName(contents[i]);
								map.add(aPlayer);
								map.WriteMapAsObjects();
								map.WriteMap();
							}
						}
					}
				}
			}
			incrementWhenShouldLook();
		}
		throw new OSProcessRelinquishControl();
	}

	/**
	  */
	public void setVerbose(boolean t) {
		this.verbose = t;
	}

	/**
	  */
	public boolean beVerbose() {
		return(this.verbose);
	}

	/**
	  */
	public void setProperties(Properties properties) {
		this.properties = properties;
	}

	/**
	  */
	public Properties getProperties() {
		return(this.properties);
	}


	/**
	  */
	StringBuffer getRoomDescription() {
		StringBuffer	rd = null;
		String			aLine = null;

		try {
			mudConnection.waitForData();
			aLine = mudConnection.readLine();
			if (rd == null) {
				rd = new StringBuffer();
			}
			while (!aLine.startsWith(mudConnection.getOutputSuffix())) {
				rd.append(aLine+"\n");
				aLine = mudConnection.readLine();
			}
		}
		catch(IOException ioe) {
		}
		return(rd);
	}

	Room lookHere() {
		StringBuffer	rd = null;
		Room		room = null;
		GregorianCalendar	now = new GregorianCalendar();

		commands.look("");
		mudConnection.findOutputPrefix();
		rd = getRoomDescription();
		if (rd != null) {
			room = roomParser.parse(rd.toString());
		}
		return(room);
	}

	/**
	  */
	StringBuffer getObjectDescription() {
		StringBuffer	od = null;
		String		aLine = null;

		try {
			mudConnection.waitForData();
			aLine = mudConnection.readLine();
			if (od == null) {
				od = new StringBuffer();
			}
			while (!aLine.startsWith(mudConnection.getOutputSuffix())) {
				od.append(aLine+"\n");
				aLine = mudConnection.readLine();
			}
		}
		catch(IOException ioe) {
		}
		return(od);
	}

	void incrementWhenShouldLook() {
		whenShouldLook.clear(Calendar.MILLISECOND);
		whenShouldLook.add(Calendar.MILLISECOND, waitTimeInterval);
	}

}
