/*
 *
 * Copyright (c) 1999, 2000 Thaddeus O. Cooper
 * 
 *          Permission is hereby granted, free of charge, to any person 
 *          obtaining a copy of this software and associated documentation
 *          files (the "Software"), to deal in the Software without 
 *          restriction, including without limitation the rights to use, copy,
 *          modify, merge, publish, distribute, sublicense, and/or sell copies 
 *          of the Software, and to permit persons to whom the Software is 
 *          furnished to do so, subject to the following conditions:
 *
 *          The above copyright notice and this permission notice shall be 
 *          included in all copies or substantial portions of the Software.
 *
 *         THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *         EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 *         MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *         NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS 
 *         BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN 
 *         ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
 *         CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE 
 *         SOFTWARE.
 *
 */
import java.io.*;
import java.util.*;
import util.*;
import os.*;
import mstar.*;
import org.place.pj.*;

/**
  * A Bot OS program to log the robot into the collaborative environment.
  *
  * @author Thaddeus O. Cooper (cooper@tweenproject.org)
  * @version 1.1
  */
public class LoginToMud extends OSProcess {
	static final String	DEFAULT_ROBOT_PROPERTIES_FILENAME="robot.properties";
	int		state = 0;
	DataInputStream	in    = null;
	FileInputStream propertiesIn = null;
	PrintStream	out   = null;
	Connection	mudConnection = null;
	Commands	commands   = null;
	Properties	properties = null;
	String		robotName = null;
	String		robotPassword = null;
	String		mudType       = null;
	boolean		verbose	      = true;	/* if set to true the bot */
						/* prints a lot of        */
						/* information to stdout. */
	String		outputPrefix  = null;
	String		outputSuffix  = null;
	String		startupCommands[] = null;

	public void initialize() {
		String		portString = null;
		String[]	args = null;
		Commandline	cl = null;
		Properties	properties = null;
		String		longOptions[] = {
							"properties",
							"name",
							"password",
							"mudtype",
							"prefix",
							"suffix",
							"startupcommands",
						};
		String		longFlags[] = {
						"verbose",
					      };
		String		startupCommandsString = null;

		args = getArgs();
		try {
			cl = new Commandline(args,
					     "nPmv",
					     "",
					     longFlags,
					     longOptions);
			// first things first, see if there is a
			// properties file that was specified
			if (cl.hasOption("properties")) {
				try {
					propertiesIn = new FileInputStream(cl.getOption("properties"));
					properties = new Properties();
					properties.load(propertiesIn);
				}
				catch (IOException ioe) {
					System.out.println("Properties file '"+
							   cl.getOption("properties")+
							   "' not found.");
					System.exit(-1);
				}
			}
			else {
				try {
					// first look in the local directory...
					propertiesIn = new FileInputStream(System.getProperty("user.dir")+
								 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
					properties = new Properties();
					properties.load(propertiesIn);
				}
				catch (IOException ioe) {
					try {
						// now try the home directory....
						propertiesIn = new FileInputStream(System.getProperty("user.home")+
									 File.separator+DEFAULT_ROBOT_PROPERTIES_FILENAME);
						properties = new Properties();
						properties.load(propertiesIn);
					}
					catch (IOException ioe1) {
					}
				}
			}
			if (properties != null) {
				verbose               = properties.getProperty("robot.verbose", "false").toLowerCase().equals("true") ? true : false;	
				robotName             = properties.getProperty("robot.name");
				robotPassword         = properties.getProperty("robot.password");
				mudType               = properties.getProperty("robot.mudType");
				outputPrefix          = properties.getProperty("robot.outputPrefix", "+++");
				outputSuffix          = properties.getProperty("robot.outputSuffix", "---");
				startupCommandsString = properties.getProperty("robot.startupCommands");
				if (startupCommandsString != null) {
					startupCommands = Utility.commaListToArray(startupCommandsString);
				}
			}

			if (cl.hasOption("v")) {
				verbose = true;
			}
			if (cl.hasOption("verbose")) {
				verbose = true;
			}
			if (cl.hasOption("n")) {
				robotName = cl.getOption("n");
			}
			if (cl.hasOption("name")) {
				robotName = cl.getOption("name");
			}
			if (cl.hasOption("P")) {
				robotPassword = cl.getOption("P");
			}
			if (cl.hasOption("password")) {
				robotPassword = cl.getOption("password");
			}
			if (cl.hasOption("m")) {
				mudType = cl.getOption("m");
			}
			if (cl.hasOption("password")) {
				mudType = cl.getOption("mudtype");
			}
			if (cl.hasOption("prefix")) {
				outputPrefix = cl.getOption("prefix");
			}
			if (cl.hasOption("suffix")) {
				outputSuffix = cl.getOption("suffix");
			}
			if (cl.hasOption("startupcommands")) {
				startupCommands = Utility.commaListToArray(cl.getOption("startupcommands"));
			}

			if (verbose) {
				if (robotName != null) {
					System.out.println("name: "+robotName);
				}
				else {
					System.out.println("name: null");
				}
				if (robotPassword != null) {
					System.out.println("password: "+robotPassword);
				}
				else {
					System.out.println("password: null");
				}
				if (mudType != null) {
					System.out.println("mudType: "+mudType);
				}
				else {
					System.out.println("mudType: null");
				}
				if (outputPrefix != null) {
					System.out.println("prefix: "+outputPrefix);
				}
				else {
					System.out.println("prefix: null");
				}
				if (outputSuffix != null) {
					System.out.println("suffix: "+outputSuffix);
				}
				else {
					System.out.println("suffix: null");
				}
				if (startupCommands != null) {
					for (int i = 0; i < startupCommands.length; i++) {
						System.out.println("startupCommands: "+startupCommands[i]);
					}
				}
				else {
					System.out.println("startupCommands: null");
				}
			}

		}
		catch (InvalidCommandlineArgument ica) {
			System.out.println("invalid command line argument "+ica.toString());
		}
		if (mudConnection == null) {
			mudConnection = (Connection)requestObject("RobotConnection");
		}
		if (commands == null) {
			commands = (Commands)requestObject("RobotCommands");
		}
		if ((mudConnection != null) && (commands != null)) {
			setInitialized(true);
		}
	}


	public void run() throws OSProcessDone {
		mudConnection.setRobotName(getRobotName());
		mudConnection.connectToMud();
		mudConnection.loginToMud(getRobotName(), getPassword());
		commands.setInputStream(mudConnection.getInputStream());
		commands.setOutputStream(mudConnection.getOutputStream());
		this.in  = mudConnection.getInputStream();
		this.out = mudConnection.getOutputStream();
		initializeSelf();
		commands.goHome();
		mudConnection.drainConnectionToSuffix();
		doStartupCommands();
		throw new OSProcessDone();
	}

	/**
	  */
	public void initializeSelf() {
		commands.setOutputPrefix(getOutputPrefix());
		commands.setOutputSuffix(getOutputSuffix());
		mudConnection.setOutputPrefix(getOutputPrefix());
		mudConnection.setOutputSuffix(getOutputSuffix());
	}

	public void doStartupCommands() {
		int	i;

		if (startupCommands != null) {
			for (i = 0; i < startupCommands.length; i++) {
				//this.out.println(startupCommands[i]);
				mudConnection.getOutputStream().println(startupCommands[i]);
				mudConnection.drainConnectionToSuffix();
			}
		}
	}

	public Properties getProperties() {
		return(this.properties);
	}

	public String getOutputSuffix() {
		return(this.outputSuffix);
	}

	public String getOutputPrefix() {
		return(this.outputPrefix);
	}

	public String getRobotName() {
		return(this.robotName);
	}

	public String getPassword() {
		return(this.robotPassword);
	}

	public String[] getStartupCommands() {
		return(this.startupCommands);
	}
}
